<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Exam;
use App\Models\Student;
use App\Models\ResultMark;
use App\Models\FinalResult;

class RecalculateExamResults extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'results:recalculate {exam_id? : The ID of the exam to recalculate (optional - leave empty for all exams)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Recalculate final results for students. Provide exam_id or leave empty to recalculate all exams.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $examId = $this->argument('exam_id');
        
        if ($examId) {
            // Recalculate single exam
            $exam = Exam::find($examId);
            
            if (!$exam) {
                $this->error("Exam with ID {$examId} not found!");
                return 1;
            }
            
            $this->recalculateExam($exam);
        } else {
            // Recalculate ALL exams
            $exams = Exam::all();
            
            if ($exams->isEmpty()) {
                $this->warn("No exams found in the database.");
                return 0;
            }
            
            $this->info("Recalculating results for ALL exams ({$exams->count()} total)...");
            
            foreach ($exams as $exam) {
                $this->newLine();
                $this->recalculateExam($exam);
            }
            
            $this->newLine();
            $this->info("✓ All exams recalculated successfully!");
        }
        
        return 0;
    }
    
    /**
     * Recalculate results for a specific exam
     */
    private function recalculateExam(Exam $exam)
    {
        $this->info("Recalculating: {$exam->name}");
        
        // Get all students who have marks for this exam
        $studentIds = ResultMark::where('exam_id', $exam->id)
            ->distinct('student_id')
            ->pluck('student_id');
        
        if ($studentIds->isEmpty()) {
            $this->warn("  No students with marks found. Skipping.");
            return;
        }
        
        $this->info("  Found " . $studentIds->count() . " students with marks.");
        
        $progressBar = $this->output->createProgressBar($studentIds->count());
        $progressBar->start();
        
        foreach ($studentIds as $studentId) {
            $student = Student::find($studentId);
            if ($student) {
                $this->calculateFinalResult($exam, $student);
            }
            $progressBar->advance();
        }
        
        $progressBar->finish();
        $this->newLine();
        $this->info("  ✓ {$exam->name} completed!");
    }

    /**
     * Calculate final result for a student (same logic as controller)
     */
    private function calculateFinalResult(Exam $exam, Student $student)
    {
        $resultMarks = ResultMark::where('exam_id', $exam->id)
            ->where('student_id', $student->id)
            ->get();

        // Identify the student's 4th/optional subject
        $optionalSubject = $student->subjects()
            ->wherePivot('is_optional', true)
            ->first();
        
        $optionalSubjectId = $optionalSubject ? $optionalSubject->id : null;

        // Separate mandatory and optional subjects
        $mandatoryMarks = $resultMarks->filter(function ($mark) use ($optionalSubjectId) {
            return $mark->subject_id !== $optionalSubjectId;
        });

        $optionalMark = $optionalSubjectId 
            ? $resultMarks->firstWhere('subject_id', $optionalSubjectId) 
            : null;

        // Check if any MANDATORY subject is failed (4th subject doesn't cause overall failure)
        $hasFailure = $mandatoryMarks->contains(function ($resultMark) {
            return $resultMark->is_failed === true;
        });

        // Calculate total marks (sum of all subjects)
        $totalMarks = $resultMarks->sum('total_marks');

        // Calculate GPA with 4th subject special rule
        if ($hasFailure) {
            $gpa = 0.00;
            $grade = 'F';
        } else {
            $mandatoryCount = $mandatoryMarks->count();
            
            if ($mandatoryCount > 0) {
                // Sum of mandatory subject GPAs
                $mandatoryGpaSum = $mandatoryMarks->sum('subject_gpa');
                
                // Add 4th subject bonus: only if GPA > 2.0, add (GPA - 2.0)
                $fourthSubjectBonus = 0;
                if ($optionalMark && $optionalMark->subject_gpa > 2.0) {
                    $fourthSubjectBonus = $optionalMark->subject_gpa - 2.0;
                }
                
                // Final GPA = (Sum of mandatory GPAs + 4th subject bonus) / Count of mandatory subjects
                $gpa = round(($mandatoryGpaSum + $fourthSubjectBonus) / $mandatoryCount, 2);
                
                // Cap GPA at 5.00 (maximum possible)
                $gpa = min($gpa, 5.00);
            } else {
                // Edge case: no mandatory subjects
                $gpa = round($resultMarks->avg('subject_gpa'), 2);
            }
            
            $grade = $this->getFinalGrade($gpa);
        }

        FinalResult::updateOrCreate(
            [
                'exam_id' => $exam->id,
                'student_id' => $student->id,
            ],
            [
                'total_marks' => $totalMarks,
                'gpa' => $gpa,
                'grade' => $grade,
            ]
        );
    }

    /**
     * Get final grade based on GPA
     */
    private function getFinalGrade($gpa)
    {
        if ($gpa >= 5.00) return 'A+';
        if ($gpa >= 4.00) return 'A';
        if ($gpa >= 3.50) return 'A-';
        if ($gpa >= 3.00) return 'B';
        if ($gpa >= 2.00) return 'C';
        if ($gpa >= 1.00) return 'D';
        return 'F';
    }
}
