<?php

namespace App\Http\Controllers;

use App\Models\ExamTemplate;
use App\Models\ExamTemplateSubjectRule;
use App\Models\Subject;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ExamTemplateController extends Controller
{
	public function index()
	{
		$templates = ExamTemplate::with('subjectRules')->latest()->paginate(15);
		return view('exam-templates.index', compact('templates'));
	}

	public function create()
	{
		$subjects = Subject::with('papers')
            ->select('id', 'name', 'code', 'type', 'has_practical')
            ->get()
            ->append(['has_mcq', 'has_cq']);
            
		return view('exam-templates.create', compact('subjects'));
	}

	public function store(Request $request)
	{
		$validated = $request->validate([
			'name' => 'required|string|max:255',
			'description' => 'nullable|string',
			'subject_rules' => 'required|array|min:1',
			'subject_rules.*.subject_id' => 'required|exists:subjects,id',
			'subject_rules.*.paper_mode' => 'required|in:merged,separate',
			'subject_rules.*.total_marks' => 'required|integer|min:0',
			'subject_rules.*.pass_marks' => 'required|integer|min:0',
			'subject_rules.*.config' => 'present',
		]);

		DB::transaction(function () use ($validated) {
			$template = ExamTemplate::create([
				'name' => $validated['name'],
				'description' => $validated['description'] ?? null,
				'academic_version_id' => \App\Models\AcademicVersion::where('is_current', true)->value('id') ?? 1,
				'created_by' => auth()->id(),
			]);

			foreach ($validated['subject_rules'] as $rule) {
				$config = $rule['config'];
				if (is_string($config)) {
					$decoded = json_decode($config, true);
					$config = is_array($decoded) ? $decoded : [];
				}
				ExamTemplateSubjectRule::create([
					'exam_template_id' => $template->id,
					'subject_id' => $rule['subject_id'],
					'paper_mode' => $rule['paper_mode'],
					'config' => $config,
					'total_marks' => $rule['total_marks'],
					'pass_marks' => $rule['pass_marks'],
				]);
			}
		});

		return redirect()->route('exam-templates.index')->with('success', 'Template created successfully');
	}

	public function edit(ExamTemplate $examTemplate)
	{
		$subjects = Subject::with('papers')
            ->select('id', 'name', 'code', 'type', 'has_practical')
            ->get()
            ->append(['has_mcq', 'has_cq']);
            
		$examTemplate->load('subjectRules.subject');
		return view('exam-templates.edit', compact('examTemplate', 'subjects'));
	}

	public function update(Request $request, ExamTemplate $examTemplate)
	{
		$validated = $request->validate([
			'name' => 'required|string|max:255',
			'description' => 'nullable|string',
			'subject_rules' => 'required|array|min:1',
			'subject_rules.*.subject_id' => 'required|exists:subjects,id',
			'subject_rules.*.paper_mode' => 'required|in:merged,separate',
			'subject_rules.*.total_marks' => 'required|integer|min:0',
			'subject_rules.*.pass_marks' => 'required|integer|min:0',
			'subject_rules.*.config' => 'present',
		]);

		DB::transaction(function () use ($examTemplate, $validated) {
			$examTemplate->update([
				'name' => $validated['name'],
				'description' => $validated['description'] ?? null,
			]);

			$examTemplate->subjectRules()->delete();
			foreach ($validated['subject_rules'] as $rule) {
				$config = $rule['config'];
				if (is_string($config)) {
					$decoded = json_decode($config, true);
					$config = is_array($decoded) ? $decoded : [];
				}
				ExamTemplateSubjectRule::create([
					'exam_template_id' => $examTemplate->id,
					'subject_id' => $rule['subject_id'],
					'paper_mode' => $rule['paper_mode'],
					'config' => $config,
					'total_marks' => $rule['total_marks'],
					'pass_marks' => $rule['pass_marks'],
				]);
			}
		});

		return redirect()->route('exam-templates.index')->with('success', 'Template updated successfully');
	}

	public function duplicate(ExamTemplate $examTemplate)
	{
		try {
			DB::beginTransaction();

			// Create a copy of the template
			$newTemplate = $examTemplate->replicate();
			$newTemplate->name = $examTemplate->name . ' (Copy)';
			$newTemplate->created_by = auth()->id();
			$newTemplate->save();

			// Copy all subject rules
			foreach ($examTemplate->subjectRules as $rule) {
				$newRule = $rule->replicate();
				$newRule->exam_template_id = $newTemplate->id;
				$newRule->save();
			}

			DB::commit();

			return redirect()->route('exam-templates.edit', $newTemplate)
				->with('success', 'Template duplicated successfully! You can now modify it.');

		} catch (\Exception $e) {
			DB::rollBack();
			return back()->with('error', 'Duplication failed: ' . $e->getMessage());
		}
	}

	public function destroy(ExamTemplate $examTemplate)
	{
		// Check if template is used in any exams
		if (\App\Models\Exam::where('exam_template_id', $examTemplate->id)->exists()) {
			return back()->with('error', 'Cannot delete this template because it is being used by one or more exams.');
		}

		// Delete associated rules first (though cascade might handle it, it's safer)
		$examTemplate->subjectRules()->delete();
		$examTemplate->delete();

		return redirect()->route('exam-templates.index')->with('success', 'Template deleted successfully');
	}
}
