<?php

namespace App\Http\Controllers;

use App\Models\HeroSlide;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class HeroSlideController extends Controller
{
    public function index()
    {
        $slides = HeroSlide::where('is_active', true)->orderBy('sort_order')->get();
        // Fallback to all if needed, but usually admin wants to see all?
        // Actually for admin index we want ALL slides.
        $slides = HeroSlide::orderBy('sort_order')->get();
        return view('hero-slides.index', compact('slides'));
    }

    public function create()
    {
        return view('hero-slides.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title' => 'nullable|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'image' => 'required|image|max:5120', // Increased max size, required for new slide
            'button_primary_text' => 'nullable|string|max:255',
            'button_primary_link' => 'nullable|string|max:255',
            'button_secondary_text' => 'nullable|string|max:255',
            'button_secondary_link' => 'nullable|string|max:255',
            'is_active' => 'nullable|boolean',
            'sort_order' => 'nullable|integer',
        ]);

        $payload = [
            'title' => $data['title'] ?? null,
            'subtitle' => $data['subtitle'] ?? null,
            'button_primary_text' => $data['button_primary_text'] ?? null,
            'button_primary_link' => $data['button_primary_link'] ?? null,
            'button_secondary_text' => $data['button_secondary_text'] ?? null,
            'button_secondary_link' => $data['button_secondary_link'] ?? null,
            'is_active' => $request->boolean('is_active'),
            'sort_order' => $data['sort_order'] ?? 0,
        ];

        if ($request->hasFile('image')) {
            // Upload to storage/app/public/hero
            $path = $request->file('image')->store('hero', 'public');
            // Save as storage/hero/filename.jpg
            $payload['image_url'] = 'storage/' . $path;
        }

        HeroSlide::create($payload);

        return redirect()->route('hero-slides.index')->with('success', 'Slide created successfully.');
    }

    public function edit(HeroSlide $hero_slide)
    {
        return view('hero-slides.edit', ['slide' => $hero_slide]);
    }

    public function update(Request $request, HeroSlide $hero_slide)
    {
        $data = $request->validate([
            'title' => 'nullable|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'image' => 'nullable|image|max:5120',
            'button_primary_text' => 'nullable|string|max:255',
            'button_primary_link' => 'nullable|string|max:255',
            'button_secondary_text' => 'nullable|string|max:255',
            'button_secondary_link' => 'nullable|string|max:255',
            'is_active' => 'nullable|boolean',
            'sort_order' => 'nullable|integer',
        ]);

        $payload = [
            'title' => $data['title'] ?? null,
            'subtitle' => $data['subtitle'] ?? null,
            'button_primary_text' => $data['button_primary_text'] ?? null,
            'button_primary_link' => $data['button_primary_link'] ?? null,
            'button_secondary_text' => $data['button_secondary_text'] ?? null,
            'button_secondary_link' => $data['button_secondary_link'] ?? null,
            'is_active' => $request->boolean('is_active'),
            'sort_order' => $data['sort_order'] ?? 0,
        ];

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($hero_slide->image_url && file_exists(public_path($hero_slide->image_url))) {
                // Check if it's a storage path or legacy public path
                $oldPath = str_replace('storage/', '', $hero_slide->image_url);
                \Illuminate\Support\Facades\Storage::disk('public')->delete($oldPath);
                
                // Also try deleting directly if it was in the old public location
                if (file_exists(public_path($hero_slide->image_url))) {
                     unlink(public_path($hero_slide->image_url));
                }
            }

            $path = $request->file('image')->store('hero', 'public');
            $payload['image_url'] = 'storage/' . $path;
        }

        $hero_slide->update($payload);

        return redirect()->route('hero-slides.index')->with('success', 'Slide updated successfully.');
    }

    public function destroy(HeroSlide $hero_slide)
    {
        // Delete image file logic
        if ($hero_slide->image_url) {
             // Handle storage path deletion
             $storagePath = str_replace('storage/', '', $hero_slide->image_url);
             if(\Illuminate\Support\Facades\Storage::disk('public')->exists($storagePath)) {
                 \Illuminate\Support\Facades\Storage::disk('public')->delete($storagePath);
             }
             
             // Handle legacy public path deletion
             if (file_exists(public_path($hero_slide->image_url))) {
                 unlink(public_path($hero_slide->image_url));
             }
        }

        $hero_slide->delete();
        return redirect()->route('hero-slides.index')->with('success', 'Slide deleted successfully.');
    }
}


