<?php

namespace App\Http\Controllers;

use App\Models\NoticeCategory;
use Illuminate\Http\Request;

class NoticeCategoryController extends Controller
{
    /**
     * Display a listing of the categories.
     */
    public function index()
    {
        $categories = NoticeCategory::ordered()->paginate(20);
        return view('notice-categories.index', compact('categories'));
    }

    /**
     * Show the form for creating a new category.
     */
    public function create()
    {
        return view('notice-categories.create');
    }

    /**
     * Store a newly created category in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:notice_categories',
            'slug' => 'nullable|string|max:255|unique:notice_categories',
            'color' => 'required|string|max:50',
            'icon' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'display_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->has('is_active');
        
        if (empty($validated['slug'])) {
            $validated['slug'] = \Str::slug($validated['name']);
        }

        NoticeCategory::create($validated);

        return redirect()->route('notice-categories.index')
            ->with('success', 'Category created successfully.');
    }

    /**
     * Show the form for editing the specified category.
     */
    public function edit(NoticeCategory $noticeCategory)
    {
        return view('notice-categories.edit', compact('noticeCategory'));
    }

    /**
     * Update the specified category in storage.
     */
    public function update(Request $request, NoticeCategory $noticeCategory)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:notice_categories,name,' . $noticeCategory->id,
            'slug' => 'nullable|string|max:255|unique:notice_categories,slug,' . $noticeCategory->id,
            'color' => 'required|string|max:50',
            'icon' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'display_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->has('is_active');
        
        if (empty($validated['slug'])) {
            $validated['slug'] = \Str::slug($validated['name']);
        }

        $noticeCategory->update($validated);

        return redirect()->route('notice-categories.index')
            ->with('success', 'Category updated successfully.');
    }

    /**
     * Remove the specified category from storage.
     */
    public function destroy(NoticeCategory $noticeCategory)
    {
        // Check if category has notices
        if ($noticeCategory->notices()->count() > 0) {
            return redirect()->route('notice-categories.index')
                ->with('error', 'Cannot delete category. It has associated notices.');
        }

        $noticeCategory->delete();

        return redirect()->route('notice-categories.index')
            ->with('success', 'Category deleted successfully.');
    }
}

