<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class SettingController extends Controller
{
    public function index()
    {
        $defaults = [
            'college_name' => 'Sheikh Abdul Wohab Model College',
            'tagline' => 'Excellence in Education',
            'description' => 'Committed to excellence in education and character development',
            'site_logo' => null,
            'site_favicon' => null,
            'controller_examination_signature' => null,
            'footer_about' => null,
            'social_facebook' => '',
            'social_youtube' => '',
            'social_twitter' => '',
            'social_instagram' => '',

            // Principal Settings
            'principal_name' => '',
            'principal_message_short' => '',
            'principal_image' => null,
            'principal_message' => '', // Full message if needed

            // Contact Page Settings
            'contact_full_address' => "Sheikh Abdul Wohab Model College\n[Your Address Here]\n[City, District, Postal Code]\nBangladesh",
            'contact_office_phone' => '+880 XXX-XXXXXXX',
            'contact_mobile_phone' => '+880 XXXX-XXXXXX',
            'contact_principal_email' => 'principal@sawmc.edu.bd',
            'contact_general_email' => 'info@sawmc.edu.bd',
            'contact_working_hours' => 'Saturday - Thursday: 9:00 AM - 4:00 PM',
            'contact_weekend_info' => 'Friday: Closed',
            'contact_map_url' => '',
            'contact_admission_title' => 'Admission Inquiries',
            'contact_admission_desc' => 'Contact us for admission procedures and requirements',
            'contact_visits_title' => 'Office Visits',
            'contact_visits_desc' => 'Welcome to visit our campus during office hours',
            'contact_support_title' => 'Support',
            'contact_support_desc' => "We're here to assist you with any queries",
        ];
        $values = [];
        foreach ($defaults as $key => $def) {
            $values[$key] = Setting::get($key, $def);
        }

        return view('settings.index', compact('values'));
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            'college_name' => 'nullable|string|max:255',
            'tagline' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'site_logo' => 'nullable|image|max:2048',
            'site_favicon' => 'nullable|image|max:1024',
            'footer_about' => 'nullable|string',
            'social_facebook' => 'nullable|url',
            'social_youtube' => 'nullable|url',
            'social_twitter' => 'nullable|url',
            'social_instagram' => 'nullable|url',

            // Principal Settings (Flat keys to match frontend)
            'principal_name' => 'nullable|string|max:255',
            'principal_message_short' => 'nullable|string',
            'principal_image' => 'nullable|image|max:2048', // Changed from url to image

            // Contact Page Settings
            'contact_full_address' => 'nullable|string',
            'contact_office_phone' => 'nullable|string|max:50',
            'contact_mobile_phone' => 'nullable|string|max:50',
            'contact_principal_email' => 'nullable|email',
            'contact_general_email' => 'nullable|email',
            'contact_working_hours' => 'nullable|string|max:255',
            'contact_weekend_info' => 'nullable|string|max:255',
            'contact_weekend_info' => 'nullable|string|max:255',
            'contact_map_url' => 'nullable|string', // Changed from url to string to allow iframe code
            'contact_admission_title' => 'nullable|string|max:255',
            'contact_admission_desc' => 'nullable|string',
            'contact_visits_title' => 'nullable|string|max:255',
            'contact_visits_desc' => 'nullable|string',
            'contact_support_title' => 'nullable|string|max:255',
            'contact_support_desc' => 'nullable|string',



            // Examination Settings
            'controller_examination_signature' => 'nullable|image|max:2048',
        ]);

        // Handle File Uploads
        $file_keys = ['site_logo', 'site_favicon', 'principal_image', 'controller_examination_signature'];
        foreach ($file_keys as $key) {
            if ($request->hasFile($key)) {
                $path = $request->file($key)->store('uploads/settings', 'public');
                $validated[$key] = $path; // Store relative path 'uploads/settings/filename.jpg'
            }
        }

        // Handle Map URL: Extract src if iframe code is pasted OR likely messy URL with quotes
        if (!empty($validated['contact_map_url'])) {
            $mapUrl = $validated['contact_map_url'];
            if (preg_match('/src="([^"]+)"/', $mapUrl, $match)) {
                $validated['contact_map_url'] = $match[1];
            } elseif (strpos($mapUrl, '"') !== false) {
                // Cut off at the first double quote if no src="" pattern found but quote exists
                $validated['contact_map_url'] = substr($mapUrl, 0, strpos($mapUrl, '"'));
            }
        }

        // Save all settings using correct groups
        foreach ($validated as $key => $value) {
            if ($key !== 'grade_rules' && !is_null($value)) {
                // Determine group based on key prefix
                $group = Str::startsWith($key, 'contact_') ? 'contact' : 'website';
                Setting::set($key, $value, $group);
            }
        }

        // Clear caches to reflect changes immediately
        \Illuminate\Support\Facades\Cache::forget('website_settings');
        \Illuminate\Support\Facades\Cache::forget('contact_info');
        \Illuminate\Support\Facades\Cache::forget('global_settings');

        return redirect()->route('settings.index')->with('success', 'Settings updated successfully.');
    }
}


