<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Result extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'student_id',
        'roll',
        'class',
        'year',
        'student_name',
        'total_marks',
        'grade',
        'remarks',
    ];

    /**
     * Get the student that owns the result.
     */
    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    /**
     * Scope a query to find result by roll, class, and year.
     */
    public function scopeFindResult($query, $roll, $class, $year)
    {
        return $query->where('roll', $roll)
                     ->where('class', $class)
                     ->where('year', $year)
                     ->first();
    }

    /**
     * Scope a query to search results.
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('student_name', 'like', "%{$search}%")
              ->orWhere('roll', 'like', "%{$search}%")
              ->orWhere('class', 'like', "%{$search}%");
        });
    }

    /**
     * Scope a query to filter by class.
     */
    public function scopeFilterByClass($query, $class)
    {
        return $query->when($class, function ($q) use ($class) {
            return $q->where('class', $class);
        });
    }

    /**
     * Scope a query to filter by year.
     */
    public function scopeFilterByYear($query, $year)
    {
        return $query->when($year, function ($q) use ($year) {
            return $q->where('year', $year);
        });
    }

    /**
     * Calculate grade based on total marks.
     */
    public static function calculateGrade($marks)
    {
        if ($marks >= 80) return 'A+';
        if ($marks >= 70) return 'A';
        if ($marks >= 60) return 'A-';
        if ($marks >= 50) return 'B';
        if ($marks >= 40) return 'C';
        if ($marks >= 33) return 'D';
        return 'F';
    }
}

