<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ResultMark extends Model
{
    use HasFactory;

    protected $fillable = [
        'exam_id',
        'student_id',
        'subject_id',
        'first_paper_mcq',
        'first_paper_cq',
        'first_paper_practical',
        'second_paper_mcq',
        'second_paper_cq',
        'second_paper_practical',
        'total_marks',
        'subject_gpa',
        'subject_grade',
        'is_failed',
    ];

    protected $casts = [
        'is_failed' => 'boolean',
    ];

    /**
     * Get the exam
     */
    public function exam()
    {
        return $this->belongsTo(Exam::class);
    }

    /**
     * Get the student
     */
    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    /**
     * Get the subject
     */
    public function subject()
    {
        return $this->belongsTo(Subject::class);
    }

    /**
     * Calculate total marks from individual components
     */
    public function calculateTotalMarks()
    {
        return ($this->first_paper_mcq ?? 0) +
            ($this->first_paper_cq ?? 0) +
            ($this->first_paper_practical ?? 0) +
            ($this->second_paper_mcq ?? 0) +
            ($this->second_paper_cq ?? 0) +
            ($this->second_paper_practical ?? 0);
    }


    /**
     * Calculate GPA based on percentage (Static Standard Rules)
     * @param float $percentage The percentage obtained (0-100)
     * @return float GPA value
     */
    public static function calculateGPAFromPercentage($percentage)
    {
        if ($percentage >= 80)
            return 5.00;
        if ($percentage >= 70)
            return 4.00;
        if ($percentage >= 60)
            return 3.50;
        if ($percentage >= 50)
            return 3.00;
        if ($percentage >= 40)
            return 2.00;
        if ($percentage >= 33)
            return 1.00;
        return 0.00;
    }

    /**
     * Get grade letter based on percentage (Static Standard Rules)
     * @param float $percentage The percentage obtained (0-100)
     * @return string Grade letter (A+, A, B, etc.)
     */
    public static function getGradeFromPercentage($percentage)
    {
        if ($percentage >= 80)
            return 'A+';
        if ($percentage >= 70)
            return 'A';
        if ($percentage >= 60)
            return 'A-';
        if ($percentage >= 50)
            return 'B';
        if ($percentage >= 40)
            return 'C';
        if ($percentage >= 33)
            return 'D';
        return 'F';
    }

    /**
     * Get complete grade data (grade, gpa, color) from percentage
     * @param float $percentage The percentage obtained (0-100)
     * @return array ['grade' => 'A+', 'gpa' => 5.0, 'color' => '#10b981']
     */
    public static function getGradeDataFromPercentage($percentage)
    {
        // Fallback to static rules logic
        $grade = self::getGradeFromPercentage($percentage);
        $gpa = self::calculateGPAFromPercentage($percentage);

        $color = match ($grade) {
            'A+', 'A' => '#10b981', // green
            'A-', 'B' => '#3b82f6', // blue
            'C' => '#f59e0b', // orange
            'D' => '#8b5cf6', // purple
            default => '#ef4444', // red
        };

        return ['grade' => $grade, 'gpa' => $gpa, 'color' => $color];
    }

    /**
     * Legacy: Calculate GPA based on total marks (DEPRECATED - use calculateGPAFromPercentage)
     * Kept for backward compatibility
     */
    public static function calculateGPA($totalMarks)
    {
        // Assume 100 marks total for legacy calls
        $percentage = ($totalMarks / 100) * 100;
        return self::calculateGPAFromPercentage($percentage);
    }

    /**
     * Legacy: Get grade based on total marks (DEPRECATED - use getGradeFromPercentage)
     * Kept for backward compatibility
     */
    public static function getGrade($totalMarks)
    {
        // Assume 100 marks total for legacy calls
        $percentage = ($totalMarks / 100) * 100;
        return self::getGradeFromPercentage($percentage);
    }
}

