<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Student extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'ssc_reg_no',
        'class_roll',
        'name',
        'father_name',
        'mother_name',
        'date_of_birth',
        'gender',
        'class',
        'group_code',
        'ssc_batch',
        'hsc_batch',
        'serial',
        'phone',
        'email',
        'address',
        'photo',
        'status',
        'batch_id', // Keeping for backward compatibility or linking to internal Batches table if needed
        'academic_version_id',
    ];

    // Relationships
    public function batch()
    {
        return $this->belongsTo(Batch::class);
    }

    public function version()
    {
        return $this->belongsTo(AcademicVersion::class, 'academic_version_id');
    }

    public function subjects()
    {
        return $this->belongsToMany(Subject::class, 'student_subjects')
            ->withPivot(['is_optional', 'academic_version_id'])
            ->withTimestamps();
    }

    public function results()
    {
        return $this->hasMany(FinalResult::class);
    }

    public function marks()
    {
        return $this->hasMany(Mark::class);
    }

    // Helpers
    public function getOptionalSubjectAttribute()
    {
        return $this->subjects()->wherePivot('is_optional', true)->first();
    }

    public function getGroupAttribute()
    {
        return match ($this->group_code) {
            3 => 'Science',
            2 => 'Commerce',
            1 => 'Humanities',
            0 => 'Humanities',
            default => 'Unknown',
        };
    }

    protected static function boot()
    {
        parent::boot();

        static::saving(function ($student) {
            // Auto-calculate HSC Batch
            if ($student->ssc_batch && empty($student->hsc_batch)) {
                $student->hsc_batch = $student->ssc_batch + 2;
            }

            // Sync with Batch Model (Create if not exists)
            if ($student->hsc_batch) {
                // Assuming Batch model name format is "HSC YYYY"
                $batchName = "HSC " . $student->hsc_batch;
                $batch = Batch::firstOrCreate(
                    ['name' => $batchName],
                    ['session' => ($student->hsc_batch - 2) . '-' . ($student->hsc_batch - 1)]
                );

                $student->batch_id = $batch->id;
            }
        });

        static::creating(function ($student) {
            // Generate Serial and Class Roll if not provided
            if (!$student->serial && $student->ssc_batch && isset($student->group_code)) {
                // Find max serial for this SSC Batch + Group
                $maxSerial = static::where('ssc_batch', $student->ssc_batch)
                    ->where('group_code', $student->group_code)
                    ->max('serial');

                $student->serial = $maxSerial ? $maxSerial + 1 : 1;
            }

            if (!$student->class_roll && $student->ssc_batch && isset($student->group_code) && $student->serial) {
                // Formula: [SSC last 2][3-digit Serial with group offset]
                // Group 1 (Humanities): 23001-23199 (serial 1-199)
                // Group 2 (Commerce): 23200-23299 (serial 1-100 -> 200-299)
                // Group 3 (Science): 23300-3399 (serial 1-100 -> 300-399)
                $sscLastTwo = substr((string) $student->ssc_batch, -2);

                // Auto-increment serial if roll collision exists
                $maxAttempts = 100; // Prevent infinite loop
                $attempts = 0;
                $finalSerial = $student->serial;

                do {
                    // Calculate actual serial with offset
                    $offsetSerial = match ($student->group_code) {
                        1 => $finalSerial,
                        2 => 200 + ($finalSerial - 1),
                        3 => 300 + ($finalSerial - 1),
                        default => $finalSerial
                    };

                    $potentialRoll = $sscLastTwo . str_pad($offsetSerial, 3, '0', STR_PAD_LEFT);

                    // Check if this roll already exists
                    $exists = static::where('class_roll', $potentialRoll)
                        ->where('id', '!=', $student->id ?? 0)
                        ->exists();

                    if (!$exists) {
                        // Found available roll
                        $student->serial = $finalSerial;
                        $student->class_roll = $potentialRoll;
                        break;
                    }

                    // Roll taken, try next serial
                    $finalSerial++;
                    $attempts++;

                } while ($attempts < $maxAttempts);

                if ($attempts >= $maxAttempts) {
                    throw new \Exception("Could not find available roll number after $maxAttempts attempts.");
                }
            }
        });
    }
}
