<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // 1. Group Subjects: Defines which subjects are available for a group in a version
        // and their default categorization (e.g., Physics is always Group Main in Science)
        Schema::create('group_subjects', function (Blueprint $table) {
            $table->id();
            $table->foreignId('academic_version_id')->constrained('academic_versions')->cascadeOnDelete();
            $table->foreignId('subject_id')->constrained()->cascadeOnDelete();
            $table->string('group_code'); // science, arts, commerce
            
            // Category defines the default role of this subject
            // core: Bangla, English (Everyone takes it)
            // group_main: Physics (Must take)
            // optional_pool: Can be chosen as 4th or elective
            $table->enum('category', ['core', 'group_main', 'optional_pool', 'fourth_pool']);
            
            $table->boolean('is_mandatory')->default(false); // Helper to quickly identify fixed subjects
            
            $table->timestamps();
            
            $table->unique(['academic_version_id', 'subject_id', 'group_code'], 'grp_sub_unique');
        });

        // 2. Subject Policies: High-level rules per group/version
        Schema::create('subject_policies', function (Blueprint $table) {
            $table->id();
            $table->foreignId('academic_version_id')->constrained('academic_versions')->cascadeOnDelete();
            $table->string('group_code');
            
            $table->integer('max_optional_count')->default(1); // e.g., Choose 1 details
            $table->integer('max_fourth_count')->default(1);
            
            $table->timestamps();
            $table->unique(['academic_version_id', 'group_code']);
        });

        // 3. Subject Buckets: Containers for selection sets (e.g., "Choose 1 from Bio/Math")
        Schema::create('subject_buckets', function (Blueprint $table) {
            $table->id();
            $table->foreignId('academic_version_id')->constrained('academic_versions')->cascadeOnDelete();
            $table->string('group_code');
            $table->string('name'); // "Science Elective A", "Humanities Pool 1"
            
            $table->integer('min_select')->default(1);
            $table->integer('max_select')->default(1);
            
            // UI helper to know if this bucket determines the 4th subject or the 3rd main
            $table->string('ui_type')->default('select'); // select, radio, checkbox
            
            $table->timestamps();
        });

        // 4. Bucket Items
        Schema::create('subject_bucket_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('subject_bucket_id')->constrained('subject_buckets')->cascadeOnDelete();
            $table->foreignId('subject_id')->constrained()->cascadeOnDelete();
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('subject_bucket_items');
        Schema::dropIfExists('subject_buckets');
        Schema::dropIfExists('subject_policies');
        Schema::dropIfExists('group_subjects');
    }
};
