<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // 1. Create subject_papers table
        Schema::create('subject_papers', function (Blueprint $table) {
            $table->id();
            $table->foreignId('subject_id')->constrained('subjects')->onDelete('cascade');
            $table->enum('paper_type', ['single', 'first', 'second'])->default('single');
            $table->string('paper_code')->unique(); // The Board Code (e.g. 101)
            $table->string('name_suffix')->nullable(); // "1st Paper"
            
            // Configuration Flags
            $table->boolean('has_cq')->default(true);
            $table->boolean('has_mcq')->default(true);
            $table->boolean('has_practical')->default(false);
            $table->integer('full_marks')->default(100);
            
            $table->timestamps();
            $table->softDeletes();
        });

        // 2. Modify subjects table
        Schema::table('subjects', function (Blueprint $table) {
            if (!Schema::hasColumn('subjects', 'has_multiple_papers')) {
                $table->boolean('has_multiple_papers')->default(false)->after('name');
            }
            if (!Schema::hasColumn('subjects', 'short_name')) {
                $table->string('short_name')->nullable()->after('name');
            }
            if (!Schema::hasColumn('subjects', 'is_4th_subject_eligible')) {
                $table->boolean('is_4th_subject_eligible')->default(true)->after('has_multiple_papers');
            }
        });

        // 3. Migrate Existing Data
        // Iterate all subjects and create a 'single' paper entry for them
        // using their existing 'code' and default config.
        $subjects = DB::table('subjects')->get();
        foreach ($subjects as $subject) {
            // Check if paper already exists (idempotency)
            $exists = DB::table('subject_papers')->where('subject_id', $subject->id)->exists();
            if (!$exists && !empty($subject->code)) {
                DB::table('subject_papers')->insert([
                    'subject_id' => $subject->id,
                    'paper_type' => 'single',
                    'paper_code' => $subject->code,
                    'name_suffix' => null,
                    'has_cq' => true, 
                    'has_mcq' => true,
                    // Assume practical false for legacy migration, or heuristics
                    'has_practical' => false, 
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('subject_papers');
        Schema::table('subjects', function (Blueprint $table) {
            $table->dropColumn(['has_multiple_papers', 'short_name', 'is_4th_subject_eligible']);
        });
    }
};
