<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Create exam_types table
        Schema::create('exam_types', function (Blueprint $table) {
            $table->id();
            $table->string('name')->unique();
            $table->text('description')->nullable();
            $table->timestamps();
        });

        // Create exams table with template integration
        Schema::create('exams', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->foreignId('exam_type_id')->constrained('exam_types')->cascadeOnDelete();
            $table->foreignId('exam_template_id')->constrained('exam_templates')->cascadeOnDelete();
            $table->enum('class', ['11', '12']);
            $table->string('year');
            $table->date('start_date')->nullable();
            $table->date('end_date')->nullable();
            $table->enum('status', ['upcoming', 'ongoing', 'completed'])->default('upcoming');
            $table->timestamps();
        });

        // Create exam_subject_rules (copied from template when exam created)
        Schema::create('exam_subject_rules', function (Blueprint $table) {
            $table->id();
            $table->foreignId('exam_id')->constrained('exams')->cascadeOnDelete();
            $table->foreignId('subject_id')->constrained('subjects')->cascadeOnDelete();
            $table->enum('paper_mode', ['merged', 'separate'])->default('merged');
            $table->json('config')->nullable();
            $table->unsignedInteger('total_marks')->default(0);
            $table->unsignedInteger('pass_marks')->default(0);
            $table->timestamps();
            
            $table->unique(['exam_id', 'subject_id']);
        });

        // Re-add exam_id to result_marks if not exists
        if (!Schema::hasColumn('result_marks', 'exam_id')) {
            Schema::table('result_marks', function (Blueprint $table) {
                $table->foreignId('exam_id')->after('id')->constrained('exams')->cascadeOnDelete();
            });
        }

        // Re-add exam_id to final_results if not exists
        if (!Schema::hasColumn('final_results', 'exam_id')) {
            Schema::table('final_results', function (Blueprint $table) {
                $table->foreignId('exam_id')->after('id')->constrained('exams')->cascadeOnDelete();
            });
        }
    }

    public function down(): void
    {
        // Drop in reverse order
        if (Schema::hasColumn('final_results', 'exam_id')) {
            Schema::table('final_results', function (Blueprint $table) {
                $table->dropForeign(['exam_id']);
                $table->dropColumn('exam_id');
            });
        }

        if (Schema::hasColumn('result_marks', 'exam_id')) {
            Schema::table('result_marks', function (Blueprint $table) {
                $table->dropForeign(['exam_id']);
                $table->dropColumn('exam_id');
            });
        }

        Schema::dropIfExists('exam_subject_rules');
        Schema::dropIfExists('exams');
        Schema::dropIfExists('exam_types');
    }
};

