<?php

namespace Database\Seeders;

use App\Models\AcademicVersion;
use App\Models\Batch;
use App\Models\Setting;
use App\Models\Subject;
use App\Models\SubjectBlueprint;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        DB::transaction(function () {
            // 1. Create Super Admin
            $admin = User::firstOrCreate(
                ['email' => 'admin@college.com'],
                [
                    'name' => 'Super Admin',
                    'password' => Hash::make('password'),
                    'role' => 'super_admin',
                    'status' => 'active',
                    'email_verified_at' => now(),
                ]
            );
            $this->command->info("Admin created: {$admin->email} / password");

            // 2. Create Initial Academic Version
            $version = AcademicVersion::firstOrCreate(
                ['name' => 'v1_2024'],
                [
                    'effective_from' => now()->startOfYear(),
                    'is_current' => true,
                    'description' => 'Initial Academic Rules 2024'
                ]
            );

            // 3. Create Initial Batch
            Batch::firstOrCreate(
                ['name' => 'HSC 2024'],
                [
                    'entry_year' => 2023,
                    'graduation_year' => 2025,
                    'is_active' => true,
                ]
            );

            // 4. Create Core Subjects
            $subjects = [
                ['code' => '101', 'name' => 'Bangla 1st Paper', 'type' => 'core'],
                ['code' => '102', 'name' => 'Bangla 2nd Paper', 'type' => 'core'],
                ['code' => '107', 'name' => 'English 1st Paper', 'type' => 'core'],
                ['code' => '108', 'name' => 'English 2nd Paper', 'type' => 'core'],
                ['code' => '275', 'name' => 'ICT', 'type' => 'core'],
                ['code' => '174', 'name' => 'Physics 1st Paper', 'type' => 'science', 'has_practical' => true],
                ['code' => '175', 'name' => 'Physics 2nd Paper', 'type' => 'science', 'has_practical' => true],
            ];

            foreach ($subjects as $sub) {
                $subject = Subject::firstOrCreate(
                    ['code' => $sub['code']],
                    $sub
                );

                // Create Default Blueprint for this version
                SubjectBlueprint::firstOrCreate(
                    [
                        'subject_id' => $subject->id,
                        'academic_version_id' => $version->id,
                    ],
                    [
                        'total_marks' => 100,
                        'pass_marks' => 33,
                        'components' => isset($sub['has_practical']) && $sub['has_practical'] 
                            ? ['cq' => 50, 'mcq' => 25, 'practical' => 25] 
                            : ['cq' => 70, 'mcq' => 30],
                        'pass_rules' => isset($sub['has_practical']) && $sub['has_practical']
                             ? ['cq' => 17, 'mcq' => 8, 'practical' => 8] 
                             : ['cq' => 23, 'mcq' => 10],
                    ]
                );
            }

            // 5. Default Settings
            $settings = [
                ['key' => 'site_name', 'value' => 'My College ERP', 'group' => 'site'],
                ['key' => 'college_address', 'value' => 'Dhaka, Bangladesh', 'group' => 'site'],
                ['key' => 'maintenance_mode', 'value' => '0', 'group' => 'system'],
            ];

            foreach ($settings as $setting) {
                Setting::firstOrCreate(['key' => $setting['key']], $setting);
            }
        });
    }
}
