<?php
/**
 * Fix Script: Recalculate total_marks from component marks
 * 
 * This script recalculates the total_marks field in result_marks table
 * by summing all component marks (mcq, cq, practical for both papers).
 * 
 * Run this from command line: php fix_totals.php
 * Or access via browser: http://localhost/admin/public/../fix_totals.php
 */

// Bootstrap Laravel
require __DIR__.'/bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ResultMark;
use App\Models\Exam;

echo "=== Fix Total Marks Script ===\n\n";

// Get the exam (you can change this to target specific exam)
$examName = 'test exam 2025';
$exam = Exam::where('name', $examName)->first();

if (!$exam) {
    echo "ERROR: Exam '$examName' not found!\n";
    echo "Available exams:\n";
    Exam::all()->each(function($e) {
        echo "  - {$e->name} (ID: {$e->id})\n";
    });
    exit(1);
}

echo "Processing Exam: {$exam->name} (ID: {$exam->id})\n";
echo "Session: {$exam->session}, Class: {$exam->class}\n\n";

// Get all result marks for this exam
$resultMarks = ResultMark::with('subject', 'student')
    ->where('exam_id', $exam->id)
    ->get();

echo "Found " . $resultMarks->count() . " result records\n\n";

$fixed = 0;
$unchanged = 0;

foreach ($resultMarks as $mark) {
    $oldTotal = $mark->total_marks;
    
    // Recalculate from components
    $newTotal = ($mark->first_paper_mcq ?? 0) +
                ($mark->first_paper_cq ?? 0) +
                ($mark->first_paper_practical ?? 0) +
                ($mark->second_paper_mcq ?? 0) +
                ($mark->second_paper_cq ?? 0) +
                ($mark->second_paper_practical ?? 0);
    
    $studentName = $mark->student->name ?? 'Unknown';
    $subjectName = $mark->subject->name ?? 'Unknown';
    
    if (abs($oldTotal - $newTotal) > 0.01) {
        echo "FIXING: {$studentName} - {$subjectName}\n";
        echo "  Old total: {$oldTotal}\n";
        echo "  New total: {$newTotal}\n";
        echo "  Components: MCQ1={$mark->first_paper_mcq} CQ1={$mark->first_paper_cq} PR1={$mark->first_paper_practical} | MCQ2={$mark->second_paper_mcq} CQ2={$mark->second_paper_cq} PR2={$mark->second_paper_practical}\n";
        
        // Recalculate GPA and Grade
        $gpa = ResultMark::calculateGPA($newTotal);
        $grade = ResultMark::getGrade($newTotal);
        
        // Get pass marks from exam rules
        $rule = \App\Models\ExamSubjectRule::where('exam_id', $exam->id)
            ->where('subject_id', $mark->subject_id)
            ->first();
        $passMarks = $rule->pass_marks ?? 33;
        $isFailed = $newTotal < $passMarks;
        
        // Update
        $mark->total_marks = $newTotal;
        $mark->subject_gpa = $gpa;
        $mark->subject_grade = $grade;
        $mark->is_failed = $isFailed;
        $mark->save();
        
        echo "  ✓ Updated! New GPA: {$gpa}, Grade: {$grade}\n\n";
        $fixed++;
    } else {
        $unchanged++;
    }
}

echo "\n=== Summary ===\n";
echo "Fixed: {$fixed} records\n";
echo "Unchanged: {$unchanged} records\n";
echo "Total: " . ($fixed + $unchanged) . " records\n\n";

if ($fixed > 0) {
    echo "Now recalculating final results...\n";
    
    // Recalculate final results for all affected students
    $students = ResultMark::where('exam_id', $exam->id)
        ->distinct('student_id')
        ->pluck('student_id');
    
    foreach ($students as $studentId) {
        $student = \App\Models\Student::find($studentId);
        if ($student) {
            echo "  Recalculating for: {$student->name}\n";
            
            $resultMarks = ResultMark::where('exam_id', $exam->id)
                ->where('student_id', $student->id)
                ->get();
            
            $hasFailure = $resultMarks->contains(fn($m) => $m->is_failed === true);
            $totalMarks = $resultMarks->sum('total_marks');
            
            if ($hasFailure) {
                $gpa = 0.00;
                $grade = 'F';
            } else {
                $gpa = round($resultMarks->avg('subject_gpa'), 2);
                $grade = getFinalGrade($gpa);
            }
            
            \App\Models\FinalResult::updateOrCreate(
                ['exam_id' => $exam->id, 'student_id' => $student->id],
                ['total_marks' => $totalMarks, 'gpa' => $gpa, 'grade' => $grade]
            );
        }
    }
    
    echo "\n✓ All done! Please check the results on the frontend.\n";
} else {
    echo "No issues found. All totals are correct!\n";
}

function getFinalGrade($gpa) {
    if ($gpa >= 5.00) return 'A+';
    if ($gpa >= 4.00) return 'A';
    if ($gpa >= 3.50) return 'A-';
    if ($gpa >= 3.00) return 'B';
    if ($gpa >= 2.00) return 'C';
    if ($gpa >= 1.00) return 'D';
    return 'F';
}
