// Main JavaScript file for common functionality

// Mobile menu toggle
document.addEventListener('DOMContentLoaded', function() {
    const mobileMenuBtn = document.getElementById('mobile-menu-btn');
    const mobileMenu = document.getElementById('mobile-menu');
    const themeToggle = document.getElementById('theme-toggle');
    const themeToggleMobile = document.getElementById('theme-toggle-mobile');

    if (mobileMenuBtn && mobileMenu) {
        mobileMenuBtn.addEventListener('click', function() {
            mobileMenu.classList.toggle('hidden');
            const expanded = this.getAttribute('aria-expanded') === 'true';
            this.setAttribute('aria-expanded', (!expanded).toString());
        });
    }

    // Theme: initialize from localStorage or OS preference
    const root = document.documentElement;
    const storedTheme = localStorage.getItem('theme');
    const prefersDark = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;
    if (storedTheme === 'dark' || (!storedTheme && prefersDark)) {
        root.classList.add('dark');
    }

    function toggleTheme() {
        root.classList.toggle('dark');
        const isDark = root.classList.contains('dark');
        localStorage.setItem('theme', isDark ? 'dark' : 'light');
    }
    if (themeToggle) themeToggle.addEventListener('click', toggleTheme);
    if (themeToggleMobile) themeToggleMobile.addEventListener('click', toggleTheme);

    // Luxury Reveal on scroll animation
    const revealEls = document.querySelectorAll('[data-reveal]');
    const revealObserver = new IntersectionObserver((entries) => {
        entries.forEach((entry, index) => {
            if (entry.isIntersecting) {
                setTimeout(() => {
                    entry.target.classList.add('revealed');
                    revealObserver.unobserve(entry.target);
                }, index * 100); // Staggered animation
            }
        });
    }, { 
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    });

    revealEls.forEach((el) => {
        revealObserver.observe(el);
    });

    // Init Swiper hero slider if present (unless deferred for dynamic config)
    if (window.Swiper && document.querySelector('.hero-swiper') && !window.INIT_HERO_AFTER_CONFIG) {
        // eslint-disable-next-line no-new
        new Swiper('.hero-swiper', {
            loop: true,
            speed: 900,
            effect: 'fade',
            autoplay: { delay: 4000, disableOnInteraction: false },
            pagination: { el: '.hero-swiper .swiper-pagination', clickable: true },
            navigation: { nextEl: '.hero-swiper .swiper-button-next', prevEl: '.hero-swiper .swiper-button-prev' }
        });
    }

    // Count-up animation for stats
    const statNumbers = document.querySelectorAll('.stat-number');
    function animateCount(el, to, duration = 1500) {
        const start = 0;
        const startTime = performance.now();
        const suffix = el.getAttribute('data-suffix') || '';
        function frame(now) {
            const progress = Math.min((now - startTime) / duration, 1);
            const eased = 1 - Math.pow(1 - progress, 3); // easeOutCubic
            const current = Math.floor(start + (to - start) * eased);
            el.textContent = current + suffix;
            if (progress < 1) requestAnimationFrame(frame);
        }
        requestAnimationFrame(frame);
    }
    if (statNumbers.length) {
        const statsObserver = new IntersectionObserver((entries) => {
            entries.forEach((entry) => {
                if (entry.isIntersecting) {
                    const el = entry.target;
                    const to = parseInt(el.getAttribute('data-count-to') || '0', 10);
                    animateCount(el, to);
                    statsObserver.unobserve(el);
                }
            });
        }, { threshold: 0.4 });
        statNumbers.forEach((el) => statsObserver.observe(el));
    }
});

// API Base URL - exposed globally
window.API_BASE_URL = window.API_BASE_URL || (window.location.origin + '/admin/public/api');

// Helper function to show loading state
function showLoading(element, message = 'Loading...') {
    if (!element) {
        console.error('showLoading: element is null');
        return;
    }
    element.innerHTML = `
        <div class="text-center py-8">
            <i class="fas fa-spinner fa-spin text-3xl text-blue-600"></i>
            <p class="text-gray-600 mt-2">${message}</p>
        </div>
    `;
}

// Helper function to show error message
function showError(element, message = 'An error occurred') {
    element.innerHTML = `
        <div class="bg-red-50 p-6 rounded-lg text-center">
            <i class="fas fa-exclamation-triangle text-red-600 text-3xl mb-2"></i>
            <p class="text-red-600">${message}</p>
        </div>
    `;
}

// Helper function to show success message
function showSuccess(element, message) {
    element.innerHTML = `
        <div class="bg-green-50 p-6 rounded-lg text-center">
            <i class="fas fa-check-circle text-green-600 text-3xl mb-2"></i>
            <p class="text-green-600">${message}</p>
        </div>
    `;
}

// Helper function to show info message
function showInfo(element, message) {
    element.innerHTML = `
        <div class="bg-blue-50 p-6 rounded-lg text-center">
            <i class="fas fa-info-circle text-blue-600 text-3xl mb-2"></i>
            <p class="text-blue-600">${message}</p>
        </div>
    `;
}

// Format date function
function formatDate(dateString) {
    const options = { year: 'numeric', month: 'long', day: 'numeric' };
    return new Date(dateString).toLocaleDateString('en-US', options);
}

// Smooth scroll for anchor links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        if (target) {
            target.scrollIntoView({
                behavior: 'smooth',
                block: 'start'
            });
        }
    });
});

