// Result Search JavaScript - Enhanced Version

document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('result-search-form');
    const resultContainer = document.getElementById('result-container');
    const searchBtn = document.getElementById('search-btn');
    const seasonSelect = document.getElementById('season');
    const classSelect = document.getElementById('class');
    const examNameSelect = document.getElementById('exam_name');
    const rollInput = document.getElementById('roll');

    // Initialize: Load sessions on page load
    loadSessions();

    // Handle session change
    seasonSelect.addEventListener('change', function() {
        const season = this.value;
        if (season) {
            // Enable class dropdown
            classSelect.disabled = false;
            classSelect.classList.remove('bg-gray-100');
            classSelect.value = '';
            // Reset exam dropdown
            resetExamDropdown();
        } else {
            // Disable class and exam dropdowns
            classSelect.disabled = true;
            classSelect.classList.add('bg-gray-100');
            classSelect.value = '';
            resetExamDropdown();
        }
    });

    // Handle class change
    classSelect.addEventListener('change', function() {
        const season = seasonSelect.value;
        const classValue = this.value;
        if (season && classValue) {
            loadExams(season, classValue);
        } else {
            resetExamDropdown();
        }
    });

    // Load available sessions
    async function loadSessions() {
        const loadingEl = document.getElementById('season-loading');
        try {
            loadingEl.classList.remove('hidden');
            const response = await fetch(`${API_BASE_URL}/sessions`);
            const data = await response.json();
            
            if (data.success && data.data && data.data.length > 0) {
                seasonSelect.innerHTML = '<option value="">Select Session</option>';
                data.data.forEach(session => {
                    const option = document.createElement('option');
                    option.value = session;
                    option.textContent = session;
                    seasonSelect.appendChild(option);
                });
            } else {
                seasonSelect.innerHTML = '<option value="">No sessions available</option>';
            }
        } catch (error) {
            console.error('Error loading sessions:', error);
            seasonSelect.innerHTML = '<option value="">Error loading sessions</option>';
        } finally {
            loadingEl.classList.add('hidden');
        }
    }

    // Load exams by session and class
    async function loadExams(session, classValue) {
        const loadingEl = document.getElementById('exam-loading');
        const emptyEl = document.getElementById('exam-empty');
        
        // Reset exam dropdown
        examNameSelect.disabled = true;
        examNameSelect.classList.add('bg-gray-100');
        examNameSelect.innerHTML = '<option value="">Loading exams...</option>';
        emptyEl.classList.add('hidden');
        
        try {
            loadingEl.classList.remove('hidden');
            const params = new URLSearchParams({
                session: session,
                class: classValue
            });
            
            const response = await fetch(`${API_BASE_URL}/exams?${params.toString()}`);
            const data = await response.json();
            
            if (data.success && data.data && data.data.length > 0) {
                examNameSelect.innerHTML = '<option value="">Select Exam</option>';
                data.data.forEach(exam => {
                    const option = document.createElement('option');
                    option.value = exam.name; // Use exam name for search
                    option.textContent = `${exam.name} (${exam.exam_type})`;
                    option.setAttribute('data-exam-id', exam.id);
                    examNameSelect.appendChild(option);
                });
                examNameSelect.disabled = false;
                examNameSelect.classList.remove('bg-gray-100');
                emptyEl.classList.add('hidden');
            } else {
                examNameSelect.innerHTML = '<option value="">No exams available</option>';
                emptyEl.classList.remove('hidden');
            }
        } catch (error) {
            console.error('Error loading exams:', error);
            examNameSelect.innerHTML = '<option value="">Error loading exams</option>';
            emptyEl.classList.remove('hidden');
        } finally {
            loadingEl.classList.add('hidden');
        }
    }

    // Reset exam dropdown
    function resetExamDropdown() {
        examNameSelect.disabled = true;
        examNameSelect.classList.add('bg-gray-100');
        examNameSelect.innerHTML = '<option value="">Select Session & Class First</option>';
        document.getElementById('exam-empty').classList.add('hidden');
    }

    form.addEventListener('submit', async function(e) {
        e.preventDefault();

        // Get form values
        const roll = rollInput.value.trim();
        const classValue = classSelect.value;
        const season = seasonSelect.value.trim();
        const examName = examNameSelect.value.trim();

        // Validate inputs
        if (!roll || !classValue || !season || !examName) {
            showError(resultContainer, 'Please fill in all fields. Make sure you have selected Session, Class, and Exam Name.');
            resultContainer.classList.remove('hidden');
            return;
        }

        // Show loading state
        searchBtn.disabled = true;
        searchBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Searching...';
        resultContainer.classList.remove('hidden');
        showLoading(resultContainer, 'Searching for result...');

        try {
            // Make API request with new parameters
            const params = new URLSearchParams({
                roll: roll,
                class: classValue,
                season: season,
                exam_name: examName
            });
            
            const response = await fetch(`${API_BASE_URL}/result?${params.toString()}`);
            const data = await response.json();

            if (data.success && data.data) {
                displayResult(data.data);
            } else {
                showResultNotFound(data.message || 'Result not found');
            }
        } catch (error) {
            console.error('Error fetching result:', error);
            showError(resultContainer, 'Unable to fetch result. Please try again later.');
        } finally {
            // Reset button state
            searchBtn.disabled = false;
            searchBtn.innerHTML = '<i class="fas fa-search mr-2"></i>Search Result';
        }
    });

    function displayResult(result) {
        // Scroll to result
        resultContainer.scrollIntoView({ behavior: 'smooth', block: 'start' });
        
        const subjectsHtml = result.subjects && result.subjects.length > 0 
            ? result.subjects.map((subject, index) => {
                // Helper to format mark or show dash
                const fmt = (val) => val !== null && val !== undefined ? formatMark(val) : '-';
                
                // Check if subject has any marks
                const hasFirstPaper = subject.first_paper_mcq !== null || subject.first_paper_cq !== null || subject.first_paper_practical !== null;
                const hasSecondPaper = subject.second_paper_mcq !== null || subject.second_paper_cq !== null || subject.second_paper_practical !== null;
                
                return `
                <tr class="border-b border-gray-200 hover:bg-gray-50 transition-colors ${subject.is_failed ? 'bg-red-50' : 'bg-white'}">
                    <td class="px-6 py-4 text-center font-medium text-gray-700">${index + 1}</td>
                    <td class="px-6 py-4">
                        <div class="font-semibold text-gray-900">${subject.subject_name || 'N/A'}</div>
                        <div class="text-xs text-gray-500 mt-1">${subject.subject_code || ''}</div>
                    </td>
                    <!-- First Paper -->
                    <td class="px-4 py-4 text-center text-sm ${subject.first_paper_mcq !== null ? 'text-gray-900 font-medium' : 'text-gray-400'}">${fmt(subject.first_paper_mcq)}</td>
                    <td class="px-4 py-4 text-center text-sm ${subject.first_paper_cq !== null ? 'text-gray-900 font-medium' : 'text-gray-400'}">${fmt(subject.first_paper_cq)}</td>
                    <td class="px-4 py-4 text-center text-sm ${subject.first_paper_practical !== null ? 'text-gray-900 font-medium' : 'text-gray-400'}">${fmt(subject.first_paper_practical)}</td>
                    <!-- Second Paper -->
                    <td class="px-4 py-4 text-center text-sm border-l border-gray-200 ${subject.second_paper_mcq !== null ? 'text-gray-900 font-medium' : 'text-gray-400'}">${fmt(subject.second_paper_mcq)}</td>
                    <td class="px-4 py-4 text-center text-sm ${subject.second_paper_cq !== null ? 'text-gray-900 font-medium' : 'text-gray-400'}">${fmt(subject.second_paper_cq)}</td>
                    <td class="px-4 py-4 text-center text-sm ${subject.second_paper_practical !== null ? 'text-gray-900 font-medium' : 'text-gray-400'}">${fmt(subject.second_paper_practical)}</td>
                    <!-- Total, GPA, Grade -->
                    <td class="px-6 py-4 text-center border-l-2 border-gray-300">
                        <span class="font-bold text-lg ${subject.is_failed ? 'text-red-600' : 'text-indigo-600'}">${formatMark(subject.total_marks)}</span>
                    </td>
                    <td class="px-6 py-4 text-center">
                        <span class="font-semibold text-gray-800">${formatMark(subject.subject_gpa)}</span>
                    </td>
                    <td class="px-6 py-4 text-center">
                        <span class="inline-flex items-center px-3 py-1.5 rounded-full text-sm font-semibold ${getGradeClass(subject.subject_grade)}">
                            ${subject.subject_grade || 'N/A'}
                        </span>
                    </td>
                </tr>
            `;
            }).join('')
            : '<tr><td colspan="10" class="px-6 py-12 text-center text-gray-500 text-lg">No subject marks available</td></tr>';

        resultContainer.innerHTML = `
            <div id="result-sheet" class="bg-white rounded-xl sm:rounded-2xl shadow-2xl overflow-hidden result-sheet border-0 print-header-compact w-full mx-auto">
                <!-- Header - Luxury Result Sheet Design -->
                <div class="relative bg-gradient-to-b from-blue-950 via-blue-900 to-indigo-950 text-white overflow-hidden print-header luxury-header">
                    <!-- Decorative Border Top -->
                    <div class="absolute top-0 left-0 right-0 h-0.5 bg-gradient-to-r from-transparent via-yellow-400 to-transparent print-decorative-line"></div>
                    
                    <!-- Decorative Pattern Background -->
                    <div class="absolute inset-0 opacity-5 print-decorative-pattern">
                        <div class="absolute inset-0" style="background-image: repeating-linear-gradient(45deg, transparent, transparent 2px, rgba(255,255,255,0.1) 2px, rgba(255,255,255,0.1) 4px);"></div>
                    </div>
                    
                    <!-- Ornamental Corners -->
                    <div class="absolute top-2 left-2 w-8 h-8 border-t-2 border-l-2 border-white/30 print-ornament print-ornament-tl"></div>
                    <div class="absolute top-2 right-2 w-8 h-8 border-t-2 border-r-2 border-white/30 print-ornament print-ornament-tr"></div>
                    <div class="absolute bottom-2 left-2 w-8 h-8 border-b-2 border-l-2 border-white/30 print-ornament print-ornament-bl"></div>
                    <div class="absolute bottom-2 right-2 w-8 h-8 border-b-2 border-r-2 border-white/30 print-ornament print-ornament-br"></div>
                    
                    <div class="relative px-4 sm:px-6 md:px-8 py-4 sm:py-6 md:py-8 text-center print-header-content">
                        <!-- Logo Section -->
                        <div class="inline-flex items-center justify-center mb-3 sm:mb-4 print-logo-container">
                            <div class="relative">
                                <div class="absolute inset-0 bg-white/10 rounded-full blur-md print-logo-glow hidden sm:block"></div>
                                <div class="absolute -inset-2 border-2 border-white/20 rounded-full print-logo-ring hidden sm:block"></div>
                                <img src="assets/img/logo.png" alt="College Logo" class="h-12 w-12 sm:h-16 sm:w-16 md:h-20 md:w-20 rounded-full border-2 border-white/50 shadow-xl object-cover relative z-10 print-logo" onerror="this.style.display='none'">
                            </div>
                        </div>
                        
                        <!-- College Name with Elegant Typography -->
                        <div class="mb-2 sm:mb-3 print-college-name-wrapper">
                            <h1 class="text-sm sm:text-base md:text-xl lg:text-2xl xl:text-3xl font-serif font-bold tracking-tight sm:tracking-wide mb-1 sm:mb-2 drop-shadow-2xl print-college-name px-2 whitespace-nowrap overflow-hidden text-ellipsis">Sheikh Abdul Wohab Model College</h1>
                        </div>
                        
                        <!-- Decorative Divider -->
                        <div class="flex items-center justify-center gap-2 sm:gap-3 mb-2 sm:mb-3 print-divider px-2">
                            <div class="h-px w-6 sm:w-8 md:w-12 bg-gradient-to-r from-transparent to-white/60 print-divider-line hidden sm:block"></div>
                            <div class="w-1.5 h-1.5 sm:w-2 sm:h-2 rounded-full bg-white/40 print-divider-dot"></div>
                            <div class="h-px flex-1 max-w-xs bg-white/40 print-divider-line"></div>
                            <div class="text-[10px] sm:text-xs font-semibold tracking-wide sm:tracking-widest uppercase opacity-80 print-certificate-text whitespace-nowrap">Academic Result Certificate</div>
                            <div class="h-px flex-1 max-w-xs bg-white/40 print-divider-line"></div>
                            <div class="w-1.5 h-1.5 sm:w-2 sm:h-2 rounded-full bg-white/40 print-divider-dot"></div>
                            <div class="h-px w-6 sm:w-8 md:w-12 bg-gradient-to-l from-transparent to-white/60 print-divider-line hidden sm:block"></div>
                        </div>
                        
                        <!-- Decorative Border Bottom -->
                        <div class="absolute bottom-0 left-0 right-0 h-0.5 bg-gradient-to-r from-transparent via-yellow-400 to-transparent print-decorative-line"></div>
                    </div>
                </div>

                <!-- Student & Exam Info - Premium Professional Layout -->
                <div class="px-4 sm:px-6 md:px-12 py-3 sm:py-4 md:py-6 bg-white print-info-section">
                    <div class="max-w-6xl mx-auto">
                        <!-- Information Grid - Compact for Print -->
                        <div class="grid grid-cols-2 gap-1.5 sm:gap-3 md:gap-4 mb-3 sm:mb-4 md:mb-6 print-info-grid">
                            <!-- Student Card -->
                            <div class="relative bg-white rounded-lg sm:rounded-xl shadow-lg border border-gray-100 overflow-hidden transition-all hover:shadow-xl print-info-card">
                                <div class="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-blue-600 to-blue-500"></div>
                                <div class="p-2 sm:p-3 md:p-5 print-card-padding">
                                    <div class="mb-1.5 sm:mb-2 md:mb-3 pb-1.5 sm:pb-2 md:pb-3 border-b border-gray-100 sm:border-b-2 print-card-header">
                                        <div class="flex items-center space-x-1 sm:space-x-2 md:space-x-3">
                                            <div class="p-1 sm:p-1.5 md:p-2 bg-blue-50 rounded print-icon-container">
                                                <i class="fas fa-user-graduate text-blue-600 text-xs sm:text-base md:text-xl print-icon"></i>
                                            </div>
                                            <h3 class="text-xs sm:text-sm md:text-lg font-bold text-gray-800 uppercase tracking-tight sm:tracking-wide print-card-title">Student Information</h3>
                                </div>
                                    </div>
                                    <div class="space-y-1 sm:space-y-1.5 md:space-y-2 print-info-content">
                                        <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center py-1 sm:py-1.5 md:py-2 border-b border-gray-100 last:border-b-0 print-info-row gap-1 sm:gap-0">
                                            <span class="text-xs sm:text-sm font-semibold text-gray-500 uppercase tracking-wide print-label">Name</span>
                                            <span class="text-sm sm:text-base font-bold text-gray-900 sm:text-right break-words print-value" title="${result.student_name || 'N/A'}">${result.student_name || 'N/A'}</span>
                                        </div>
                                        <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center py-1 sm:py-1.5 md:py-2 border-b border-gray-100 last:border-b-0 print-info-row gap-1 sm:gap-0">
                                            <span class="text-xs sm:text-sm font-semibold text-gray-500 uppercase tracking-wide print-label">Roll No</span>
                                            <span class="text-sm sm:text-base font-bold text-blue-600 sm:flex-1 sm:text-right print-value">${result.roll || 'N/A'}</span>
                                        </div>
                                        <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center py-1 sm:py-1.5 md:py-2 border-b border-gray-100 last:border-b-0 print-info-row gap-1 sm:gap-0">
                                            <span class="text-xs sm:text-sm font-semibold text-gray-500 uppercase tracking-wide print-label">Class</span>
                                            <span class="text-sm sm:text-base font-bold text-gray-900 sm:flex-1 sm:text-right print-value">Class ${result.class || 'N/A'}</span>
                                    </div>
                                        <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center py-1 sm:py-1.5 md:py-2 print-info-row gap-1 sm:gap-0">
                                            <span class="text-xs sm:text-sm font-semibold text-gray-500 uppercase tracking-wide print-label">Session</span>
                                            <span class="text-sm sm:text-base font-bold text-gray-900 sm:flex-1 sm:text-right print-value">${result.session || 'N/A'}</span>
                                    </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Exam Card -->
                            <div class="relative bg-white rounded-lg sm:rounded-xl shadow-lg border border-gray-100 overflow-hidden transition-all hover:shadow-xl print-info-card">
                                <div class="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-emerald-600 to-emerald-500"></div>
                                <div class="p-2 sm:p-4 md:p-7 print-card-padding">
                                    <div class="mb-2 sm:mb-4 md:mb-6 pb-2 sm:pb-3 md:pb-4 border-b border-gray-100 sm:border-b-2 print-card-header">
                                        <div class="flex items-center space-x-1 sm:space-x-2 md:space-x-3">
                                            <div class="p-1 sm:p-1.5 md:p-2 bg-emerald-50 rounded print-icon-container">
                                                <i class="fas fa-file-alt text-emerald-600 text-xs sm:text-base md:text-xl print-icon"></i>
                                            </div>
                                            <h3 class="text-xs sm:text-sm md:text-lg font-bold text-gray-800 uppercase tracking-tight sm:tracking-wide print-card-title">Examination Details</h3>
                                </div>
                                    </div>
                                    <div class="space-y-1 sm:space-y-1.5 md:space-y-2 print-info-content">
                                        <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center py-1 sm:py-1.5 md:py-2 border-b border-gray-100 last:border-b-0 print-info-row gap-1 sm:gap-0">
                                            <span class="text-xs sm:text-sm font-semibold text-gray-500 uppercase tracking-wide print-label">Exam</span>
                                            <span class="text-sm sm:text-base font-bold text-gray-900 sm:flex-1 sm:text-right break-words print-value" title="${result.exam_name || 'N/A'}">${result.exam_name || 'N/A'}</span>
                                        </div>
                                        <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center py-1 sm:py-1.5 md:py-2 border-b border-gray-100 last:border-b-0 print-info-row gap-1 sm:gap-0">
                                            <span class="text-xs sm:text-sm font-semibold text-gray-500 uppercase tracking-wide print-label">Type</span>
                                            <span class="text-sm sm:text-base font-bold text-gray-900 sm:flex-1 sm:text-right print-value">${result.exam_type || 'N/A'}</span>
                                        </div>
                                        <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center py-1 sm:py-1.5 md:py-2 print-info-row gap-1 sm:gap-0">
                                            <span class="text-xs sm:text-sm font-semibold text-gray-500 uppercase tracking-wide print-label">Position</span>
                                            <span class="text-lg sm:text-xl md:text-2xl font-extrabold text-gradient bg-gradient-to-r from-blue-600 to-indigo-600 bg-clip-text text-transparent sm:flex-1 sm:text-right print-value print-position">${result.position ? `#${result.position}` : 'N/A'}</span>
                                    </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Summary Metrics - Compact Single Line Design -->
                        <div class="relative bg-gray-50 rounded-lg p-3 sm:p-4 shadow-sm border border-gray-200 print-summary-section">
                            <div class="absolute top-0 right-0 w-32 h-32 bg-blue-200/10 rounded-full -mr-16 -mt-16 blur-2xl print-hide"></div>
                            <div class="relative">
                                <h3 class="text-center text-xs sm:text-sm font-bold text-gray-800 uppercase tracking-wide mb-2 sm:mb-3 print-summary-title">Final Result Summary</h3>
                                <div class="flex flex-nowrap items-center justify-center gap-1 sm:gap-2 md:gap-3 lg:gap-4 print-summary-grid">
                                <!-- Total Marks -->
                                    <div class="flex items-center space-x-0.5 sm:space-x-1 md:space-x-1.5 print-summary-item whitespace-nowrap">
                                        <span class="text-[8px] sm:text-[10px] md:text-xs font-bold text-gray-600 uppercase tracking-tight sm:tracking-wide print-summary-label">Total</span>
                                        <span class="text-xs sm:text-sm md:text-base lg:text-lg font-extrabold text-blue-600 print-summary-value">${formatMark(result.total_marks)}</span>
                                </div>
                                    <!-- Separator -->
                                    <div class="h-4 sm:h-6 w-px bg-gray-300 print-summary-separator"></div>
                                <!-- GPA -->
                                    <div class="flex items-center space-x-0.5 sm:space-x-1 md:space-x-1.5 print-summary-item whitespace-nowrap">
                                        <span class="text-[8px] sm:text-[10px] md:text-xs font-bold text-gray-600 uppercase tracking-tight sm:tracking-wide print-summary-label">GPA</span>
                                        <span class="text-xs sm:text-sm md:text-base lg:text-lg font-extrabold text-indigo-600 print-summary-value">${formatMark(result.gpa)}</span>
                                </div>
                                    <!-- Separator -->
                                    <div class="h-4 sm:h-6 w-px bg-gray-300 print-summary-separator"></div>
                                <!-- Grade -->
                                    <div class="flex items-center space-x-0.5 sm:space-x-1 md:space-x-1.5 print-summary-item whitespace-nowrap">
                                        <span class="text-[8px] sm:text-[10px] md:text-xs font-bold text-gray-600 uppercase tracking-tight sm:tracking-wide print-summary-label">Grade</span>
                                        <span class="text-xs sm:text-sm md:text-base lg:text-lg font-extrabold text-purple-600 print-summary-value">${result.grade || 'N/A'}</span>
                                </div>
                                    <!-- Separator -->
                                    <div class="h-4 sm:h-6 w-px bg-gray-300 print-summary-separator"></div>
                                <!-- Status -->
                                    <div class="flex items-center space-x-0.5 sm:space-x-1 md:space-x-1.5 print-summary-item whitespace-nowrap">
                                        <span class="text-[8px] sm:text-[10px] md:text-xs font-bold text-gray-600 uppercase tracking-tight sm:tracking-wide print-summary-label">Status</span>
                                        <span class="text-[10px] sm:text-xs md:text-sm font-extrabold ${result.grade && result.grade.toUpperCase() !== 'F' ? 'text-emerald-600' : 'text-red-600'} print-summary-value">${getStatus(result.grade)}</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Subject Marks Table - Premium Professional Design -->
                ${result.subjects && result.subjects.length > 0 ? `
                <div class="px-2 sm:px-6 md:px-12 pb-3 sm:pb-8 md:pb-12 bg-gray-50">
                    <div class="max-w-6xl mx-auto">
                        <div class="mb-1.5 sm:mb-6 md:mb-8 text-center">
                            <div class="inline-flex items-center space-x-1 sm:space-x-3 mb-1 sm:mb-3 md:mb-4">
                                <div class="h-0.5 sm:h-1 w-4 sm:w-8 md:w-12 bg-gradient-to-r from-blue-600 to-indigo-600 rounded-full"></div>
                                <h3 class="text-xs sm:text-lg md:text-xl lg:text-2xl font-extrabold text-gray-800 uppercase tracking-tight sm:tracking-wide px-1 sm:px-2">Subject-Wise Marks & Grades</h3>
                                <div class="h-0.5 sm:h-1 w-4 sm:w-8 md:w-12 bg-gradient-to-r from-indigo-600 to-blue-600 rounded-full"></div>
                            </div>
                            <p class="text-[9px] sm:text-sm text-gray-500 px-1 sm:px-2">Detailed breakdown of academic performance</p>
                            <p class="text-[8px] sm:hidden text-gray-400 px-1 mt-1 italic">← Swipe to see all columns →</p>
                        </div>
                        <div class="overflow-x-auto sm:overflow-x-visible shadow-2xl rounded-lg sm:rounded-xl border border-gray-200 -mx-2 sm:-mx-4 md:mx-0 mobile-table-container">
                            <table class="w-full border-collapse bg-white result-marks-table mobile-marks-table min-w-[900px] sm:min-w-full">
                                <thead>
                                    <!-- First Header Row -->
                                    <tr class="bg-gradient-to-r from-blue-600 via-blue-600 to-blue-700">
                                        <th rowspan="3" class="px-1 sm:px-2 md:px-4 py-1 sm:py-2 md:py-4 text-[9px] sm:text-xs md:text-sm font-bold text-white uppercase border-r border-white/20 text-center align-middle">Subject Code</th>
                                        <th rowspan="3" class="px-1 sm:px-2 md:px-5 py-1 sm:py-2 md:py-4 text-[9px] sm:text-xs md:text-sm font-bold text-white uppercase border-r border-white/20 text-left align-middle">Subject Name</th>
                                        <th colspan="6" class="px-1 sm:px-2 md:px-4 py-1 sm:py-2 md:py-4 text-[9px] sm:text-xs md:text-sm font-bold text-white uppercase border-r border-white/20 text-center">Marks Breakdown</th>
                                        <th rowspan="3" class="px-1 sm:px-2 md:px-4 py-1 sm:py-2 md:py-4 text-[9px] sm:text-xs md:text-sm font-bold text-white uppercase border-r border-white/20 text-center align-middle">Total</th>
                                        <th rowspan="3" class="px-1 sm:px-2 md:px-4 py-1 sm:py-2 md:py-4 text-[9px] sm:text-xs md:text-sm font-bold text-white uppercase text-center align-middle">Grade</th>
                                    </tr>
                                    <!-- Second Header Row - Paper Division -->
                                    <tr class="bg-gradient-to-r from-blue-600 via-blue-600 to-blue-700">
                                        <th colspan="3" class="px-1 sm:px-2 md:px-4 py-1 sm:py-1.5 md:py-2 text-[8px] sm:text-[10px] md:text-xs font-semibold text-white uppercase border-r border-white/20 text-center">1st Paper</th>
                                        <th colspan="3" class="px-1 sm:px-2 md:px-4 py-1 sm:py-1.5 md:py-2 text-[8px] sm:text-[10px] md:text-xs font-semibold text-white uppercase border-r border-white/20 text-center">2nd Paper</th>
                                    </tr>
                                    <!-- Third Header Row - Mark Types -->
                                    <tr class="bg-gradient-to-r from-blue-600 via-blue-600 to-blue-700">
                                        <th class="px-1 sm:px-2 md:px-3 py-1 sm:py-1.5 md:py-2 text-[8px] sm:text-[9px] md:text-xs font-semibold text-white uppercase border-r border-white/20 text-center">CQ</th>
                                        <th class="px-1 sm:px-2 md:px-3 py-1 sm:py-1.5 md:py-2 text-[8px] sm:text-[9px] md:text-xs font-semibold text-white uppercase border-r border-white/20 text-center">MCQ</th>
                                        <th class="px-1 sm:px-2 md:px-3 py-1 sm:py-1.5 md:py-2 text-[8px] sm:text-[9px] md:text-xs font-semibold text-white uppercase border-r border-white/20 text-center">Prac</th>
                                        <th class="px-1 sm:px-2 md:px-3 py-1 sm:py-1.5 md:py-2 text-[8px] sm:text-[9px] md:text-xs font-semibold text-white uppercase border-r border-white/20 text-center">CQ</th>
                                        <th class="px-1 sm:px-2 md:px-3 py-1 sm:py-1.5 md:py-2 text-[8px] sm:text-[9px] md:text-xs font-semibold text-white uppercase border-r border-white/20 text-center">MCQ</th>
                                        <th class="px-1 sm:px-2 md:px-3 py-1 sm:py-1.5 md:py-2 text-[8px] sm:text-[9px] md:text-xs font-semibold text-white uppercase border-r border-white/20 text-center">Prac</th>
                                    </tr>
                                </thead>
                            <tbody>
                                ${result.subjects.map((subject, index) => {
                                    // Helper to format mark or show dash for empty
                                    const fmt = (val) => {
                                        if (val === null || val === undefined) return '-';
                                        const num = parseFloat(val);
                                        if (isNaN(num)) return '-';
                                        if (num === 0) return '0';
                                        if (num % 1 === 0) return num.toString();
                                        return num.toFixed(2);
                                    };
                                    
                                    // Debug logging (uncomment if needed for troubleshooting)
                                    // console.log('=== Subject:', subject.subject_name, '===');
                                    // console.log('DATABASE VALUES:');
                                    // console.log('  first_paper_mcq:', subject.first_paper_mcq);
                                    // console.log('  first_paper_cq:', subject.first_paper_cq);
                                    // console.log('  first_paper_practical:', subject.first_paper_practical);
                                    // console.log('  second_paper_mcq:', subject.second_paper_mcq);
                                    // console.log('  second_paper_cq:', subject.second_paper_cq);
                                    // console.log('  second_paper_practical:', subject.second_paper_practical);
                                    // console.log('  STORED total_marks:', subject.total_marks);
                                    // console.log('DISPLAY CALCULATION:');
                                    // console.log('  Theory (CQ sum):', theory);
                                    // console.log('  Practical (Practical sum):', practical);
                                    // console.log('  MCQ (MCQ sum):', mcq);
                                    // console.log('  Calculated Total:', (theory + practical + mcq));
                                    // console.log('---');
                                    
                                    
                                    return `
                                <tr class="${index % 2 === 0 ? 'bg-white' : 'bg-gray-50'} hover:bg-blue-50 transition-colors">
                                    <td class="px-1 sm:px-2 md:px-4 py-2 sm:py-2 md:py-3 text-[9px] sm:text-xs md:text-sm text-center font-semibold text-gray-700 border-r border-b border-gray-200">${subject.subject_code || 'N/A'}</td>
                                    <td class="px-1 sm:px-2 md:px-5 py-2 sm:py-2 md:py-3 text-[9px] sm:text-xs md:text-sm text-left font-medium text-gray-900 border-r border-b border-gray-200">
                                        <span class="break-words">${subject.subject_name || 'N/A'}</span>
                                    </td>
                                    <!-- 1st Paper Marks -->
                                    <td class="px-1 sm:px-2 md:px-3 py-2 sm:py-2 md:py-3 text-[9px] sm:text-xs md:text-sm text-center font-medium text-gray-800 border-r border-b border-gray-200 whitespace-nowrap">${fmt(subject.first_paper_cq)}</td>
                                    <td class="px-1 sm:px-2 md:px-3 py-2 sm:py-2 md:py-3 text-[9px] sm:text-xs md:text-sm text-center font-medium text-gray-800 border-r border-b border-gray-200 whitespace-nowrap">${fmt(subject.first_paper_mcq)}</td>
                                    <td class="px-1 sm:px-2 md:px-3 py-2 sm:py-2 md:py-3 text-[9px] sm:text-xs md:text-sm text-center font-medium text-gray-800 border-r border-b border-gray-200 whitespace-nowrap">${fmt(subject.first_paper_practical)}</td>
                                    <!-- 2nd Paper Marks -->
                                    <td class="px-1 sm:px-2 md:px-3 py-2 sm:py-2 md:py-3 text-[9px] sm:text-xs md:text-sm text-center font-medium text-gray-800 border-r border-b border-gray-200 whitespace-nowrap">${fmt(subject.second_paper_cq)}</td>
                                    <td class="px-1 sm:px-2 md:px-3 py-2 sm:py-2 md:py-3 text-[9px] sm:text-xs md:text-sm text-center font-medium text-gray-800 border-r border-b border-gray-200 whitespace-nowrap">${fmt(subject.second_paper_mcq)}</td>
                                    <td class="px-1 sm:px-2 md:px-3 py-2 sm:py-2 md:py-3 text-[9px] sm:text-xs md:text-sm text-center font-medium text-gray-800 border-r border-b border-gray-200 whitespace-nowrap">${fmt(subject.second_paper_practical)}</td>
                                    <!-- Total & Grade -->
                                    <td class="px-1 sm:px-2 md:px-4 py-2 sm:py-2 md:py-3 text-xs sm:text-sm md:text-base lg:text-lg text-center font-bold border-r border-b border-gray-200 text-gray-900">${formatMark(subject.total_marks)}</td>
                                    <td class="px-1 sm:px-2 md:px-4 py-2 sm:py-2 md:py-3 text-center border-b border-gray-200">
                                        <span class="inline-flex items-center justify-center min-w-[1.5rem] sm:min-w-[2rem] md:min-w-[2.5rem] px-1 sm:px-2 py-0.5 sm:py-1 rounded-md font-bold text-[9px] sm:text-xs md:text-sm ${subject.subject_grade === 'A+' ? 'bg-green-100 text-green-700' : subject.subject_grade === 'A' ? 'bg-blue-100 text-blue-700' : subject.subject_grade === 'B' ? 'bg-yellow-100 text-yellow-700' : subject.subject_grade === 'F' ? 'bg-red-100 text-red-700' : 'bg-gray-100 text-gray-700'}">${subject.subject_grade || 'N/A'}</span>
                                    </td>
                                </tr>
                                `;
                                }).join('')}
                            </tbody>
                        </table>
                    </div>
                </div>
                ` : ''}

                <!-- Action Buttons -->
                <div class="bg-gradient-to-r from-gray-50 to-gray-100 px-4 sm:px-6 md:px-12 py-4 sm:py-6 md:py-8 border-t border-gray-200 flex flex-wrap justify-center gap-3 sm:gap-4 no-print">
                    <button onclick="printMarksheet()" class="group relative bg-gradient-to-r from-emerald-600 to-emerald-700 hover:from-emerald-700 hover:to-emerald-800 text-white px-4 sm:px-6 md:px-8 py-2.5 sm:py-3 rounded-lg sm:rounded-xl transition-all duration-300 shadow-lg hover:shadow-2xl font-semibold text-xs sm:text-sm flex items-center justify-center w-full sm:w-auto transform hover:-translate-y-0.5">
                        <i class="fas fa-print mr-2 group-hover:animate-pulse"></i>
                        <span>Print</span>
                    </button>
                    <button onclick="location.reload()" class="group relative bg-gradient-to-r from-gray-600 to-gray-700 hover:from-gray-700 hover:to-gray-800 text-white px-4 sm:px-6 md:px-8 py-2.5 sm:py-3 rounded-lg sm:rounded-xl transition-all duration-300 shadow-lg hover:shadow-2xl font-semibold text-xs sm:text-sm flex items-center justify-center w-full sm:w-auto transform hover:-translate-y-0.5">
                        <i class="fas fa-search mr-2 group-hover:rotate-90 transition-transform"></i>
                        <span>New Search</span>
                    </button>
                </div>
            </div>
        `;

        // Store result data globally
        window.currentResult = result;
        
        // Auto-fit content for print after a short delay to ensure rendering
        setTimeout(() => {
            optimizeForPrint();
        }, 100);
    }
    
    // Auto-fit result to single A4 page for printing
    function optimizeForPrint() {
        const resultSheet = document.getElementById('result-sheet');
        if (!resultSheet) return;
        
        // Count subjects to determine scaling needs
        const subjectCount = resultSheet.querySelectorAll('.result-marks-table tbody tr').length;
        
        // Apply compact styling for many subjects
        if (subjectCount > 12) {
            resultSheet.classList.add('print-many-subjects');
            
            // Reduce table font size and padding
            const table = resultSheet.querySelector('.result-marks-table');
            if (table) {
                table.style.fontSize = '7px';
                table.querySelectorAll('th, td').forEach(cell => {
                    cell.style.padding = '1mm 0.8mm';
                    cell.style.fontSize = '7px';
                });
            }
            
            // Reduce spacing in summary section
            const summarySection = resultSheet.querySelector('.bg-gradient-to-br.from-blue-50');
            if (summarySection) {
                summarySection.style.padding = '2mm';
                summarySection.style.marginBottom = '2mm';
            }
            
            // Reduce header padding
            const header = resultSheet.querySelector('.bg-gradient-to-br');
            if (header) {
                header.style.padding = '4mm 3mm';
            }
        } else if (subjectCount > 8) {
            resultSheet.classList.add('print-moderate-subjects');
            
            // Moderate reduction
            const table = resultSheet.querySelector('.result-marks-table');
            if (table) {
                table.style.fontSize = '7.5px';
                table.querySelectorAll('th, td').forEach(cell => {
                    cell.style.padding = '1.2mm 1mm';
                });
            }
        }
    }

    function showResultNotFound(message) {
        resultContainer.innerHTML = `
            <div class="bg-white rounded-lg shadow-lg overflow-hidden">
                <div class="bg-gradient-to-r from-red-600 to-red-500 text-white p-6 text-center">
                    <i class="fas fa-times-circle text-5xl mb-3"></i>
                    <h3 class="text-2xl font-bold">Result Not Found</h3>
                </div>
                <div class="p-8 text-center">
                    <p class="text-gray-600 mb-6">${message || 'No result found matching your search criteria. Please verify the following:'}</p>
                    <ul class="text-left max-w-md mx-auto space-y-2 mb-6">
                        <li class="flex items-start">
                            <i class="fas fa-check text-blue-600 mr-2 mt-1"></i>
                            <span class="text-gray-700">Roll number is correct</span>
                        </li>
                        <li class="flex items-start">
                            <i class="fas fa-check text-blue-600 mr-2 mt-1"></i>
                            <span class="text-gray-700">Class selection is accurate</span>
                        </li>
                        <li class="flex items-start">
                            <i class="fas fa-check text-blue-600 mr-2 mt-1"></i>
                            <span class="text-gray-700">Session format is correct (YYYY-YYYY)</span>
                        </li>
                        <li class="flex items-start">
                            <i class="fas fa-check text-blue-600 mr-2 mt-1"></i>
                            <span class="text-gray-700">Exam name matches exactly</span>
                        </li>
                        <li class="flex items-start">
                            <i class="fas fa-check text-blue-600 mr-2 mt-1"></i>
                            <span class="text-gray-700">Results have been published for this exam</span>
                        </li>
                    </ul>
                    <button onclick="location.reload()" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition">
                        <i class="fas fa-redo mr-2"></i>Try Again
                    </button>
                </div>
            </div>
        `;
    }

    function formatMark(mark) {
        if (mark === null || mark === undefined) return '-';
        const num = parseFloat(mark);
        // If it's a whole number, don't show decimals
        if (num % 1 === 0) {
            return num.toString();
        }
        // Otherwise show 2 decimal places
        return num.toFixed(2);
    }

    function getStatus(grade) {
        if (!grade) return 'N/A';
        const failGrades = ['F', 'FAIL'];
        return failGrades.includes(grade.toUpperCase()) ? 'Failed' : 'Passed';
    }

    function getStatusColor(grade) {
        const failGrades = ['F', 'FAIL'];
        return failGrades.includes((grade || '').toUpperCase()) 
            ? 'from-red-500 to-red-600' 
            : 'from-green-500 to-green-600';
    }

    function getGradeClass(grade) {
        if (!grade) return 'bg-gray-200 text-gray-700';
        const gradeUpper = grade.toUpperCase();
        if (gradeUpper === 'A+') return 'bg-yellow-100 text-yellow-800';
        if (gradeUpper === 'A') return 'bg-green-100 text-green-800';
        if (gradeUpper === 'A-') return 'bg-blue-100 text-blue-800';
        if (gradeUpper === 'B') return 'bg-purple-100 text-purple-800';
        if (gradeUpper === 'C') return 'bg-orange-100 text-orange-800';
        if (gradeUpper === 'D') return 'bg-pink-100 text-pink-800';
        if (gradeUpper === 'F') return 'bg-red-100 text-red-800';
        return 'bg-gray-200 text-gray-700';
    }
});

// Print Marksheet Only (A4 Single Page)
function printMarksheet() {
    window.print();
}


// Print styles are handled in CSS file (style.css)