// Teachers page JavaScript

// API Base URL
const API_BASE_URL = window.API_BASE_URL || (window.location.origin + '/admin/public/api');

document.addEventListener('DOMContentLoaded', function() {
    loadTeachers();
});

async function loadTeachers() {
    const container = document.getElementById('teachers-container');
    const loadingState = document.getElementById('loading-state');
    
    try {
        const response = await fetch(`${API_BASE_URL}/teachers`);
        const data = await response.json();

        if (data.success && data.data && data.data.length > 0) {
            loadingState.style.display = 'none';
            displayTeachers(data.data, container);
        } else {
            loadingState.innerHTML = `
                <div class="bg-white rounded-2xl shadow-xl p-12 text-center">
                    <div class="w-20 h-20 mx-auto mb-4 rounded-full bg-blue-100 flex items-center justify-center">
                        <i class="fas fa-users text-3xl text-blue-600"></i>
                    </div>
                    <p class="text-gray-600 text-lg">No teachers information available at the moment.</p>
                </div>
            `;
        }
    } catch (error) {
        console.error('Error fetching teachers:', error);
        loadingState.innerHTML = `
            <div class="bg-red-50 rounded-2xl shadow-xl p-12 text-center border border-red-200">
                <div class="w-20 h-20 mx-auto mb-4 rounded-full bg-red-100 flex items-center justify-center">
                    <i class="fas fa-exclamation-triangle text-3xl text-red-600"></i>
                </div>
                <p class="text-red-600 font-semibold text-lg">Unable to load teachers. Please try again later.</p>
            </div>
        `;
    }
}

function displayTeachers(teachers, container) {
    // Separate principal from other teachers
    const principal = teachers.find(t => t.is_principal === true || t.is_principal === 1);
    const otherTeachers = teachers.filter(t => !t.is_principal || t.is_principal === 0);

    let html = '';

    // Principal Section (Highlighted)
    if (principal) {
        html += `
            <div class="mb-16">
                <div class="text-center mb-8">
                    <h3 class="text-3xl md:text-4xl font-bold text-gray-800 mb-2">Principal</h3>
                    <div class="w-24 h-1 bg-gradient-to-r from-transparent via-blue-600 to-transparent mx-auto"></div>
                </div>
                <div class="flex justify-center">
                    ${createPrincipalCard(principal)}
                </div>
            </div>
        `;
    }

    // Other Teachers Section
    if (otherTeachers.length > 0) {
        html += `
            <div>
                <div class="text-center mb-12">
                    <h3 class="text-3xl md:text-4xl font-bold text-gray-800 mb-2">Faculty Members</h3>
                    <div class="w-24 h-1 bg-gradient-to-r from-transparent via-blue-600 to-transparent mx-auto"></div>
                </div>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                    ${otherTeachers.map(teacher => createTeacherCard(teacher)).join('')}
                </div>
            </div>
        `;
    }

    container.innerHTML = html;
}

function createPrincipalCard(teacher) {
    const photoUrl = teacher.photo ? (window.location.origin + '/admin/public/storage/' + teacher.photo) : 'assets/img/default-teacher.png';
    
    return `
        <div class="w-full max-w-2xl bg-gradient-to-br from-blue-50 via-white to-indigo-50 rounded-3xl shadow-2xl overflow-hidden border-2 border-blue-200 transform hover:scale-[1.02] transition-all duration-300">
            <div class="bg-gradient-to-r from-blue-600 via-blue-700 to-indigo-700 p-8 text-center">
                <div class="relative inline-block">
                    <div class="absolute inset-0 bg-white/30 rounded-full blur-xl"></div>
                    <img src="${photoUrl}" 
                         alt="${teacher.name}" 
                         onerror="this.src='assets/img/default-teacher.png'"
                         class="relative w-32 h-32 rounded-full border-4 border-white shadow-2xl object-cover mx-auto">
                    <div class="absolute -bottom-2 -right-2 bg-gradient-to-r from-yellow-400 to-yellow-500 rounded-full p-3 shadow-lg">
                        <i class="fas fa-crown text-white text-xl"></i>
                    </div>
                </div>
                <h4 class="text-2xl md:text-3xl font-bold text-white mt-6 mb-2">${teacher.name}</h4>
                <p class="text-blue-100 text-lg font-semibold">${teacher.designation || 'Principal'}</p>
            </div>
            <div class="p-8">
                ${teacher.qualification ? `
                    <div class="mb-4">
                        <div class="flex items-start">
                            <i class="fas fa-graduation-cap text-blue-600 text-xl mr-3 mt-1"></i>
                            <div>
                                <p class="text-sm font-semibold text-gray-700 mb-1">Qualification</p>
                                <p class="text-gray-600">${teacher.qualification}</p>
                            </div>
                        </div>
                    </div>
                ` : ''}
                ${teacher.specialization ? `
                    <div class="mb-4">
                        <div class="flex items-start">
                            <i class="fas fa-book text-blue-600 text-xl mr-3 mt-1"></i>
                            <div>
                                <p class="text-sm font-semibold text-gray-700 mb-1">Specialization</p>
                                <p class="text-gray-600">${teacher.specialization}</p>
                            </div>
                        </div>
                    </div>
                ` : ''}
                ${teacher.bio ? `
                    <div class="mb-4">
                        <p class="text-gray-700 leading-relaxed">${teacher.bio}</p>
                    </div>
                ` : ''}
                <div class="flex flex-wrap gap-4 mt-6 pt-6 border-t border-gray-200">
                    ${teacher.email ? `
                        <a href="mailto:${teacher.email}" class="flex items-center text-blue-600 hover:text-blue-800 transition">
                            <i class="fas fa-envelope mr-2"></i>
                            <span class="text-sm">${teacher.email}</span>
                        </a>
                    ` : ''}
                    ${teacher.phone ? `
                        <a href="tel:${teacher.phone}" class="flex items-center text-blue-600 hover:text-blue-800 transition">
                            <i class="fas fa-phone mr-2"></i>
                            <span class="text-sm">${teacher.phone}</span>
                        </a>
                    ` : ''}
                </div>
            </div>
        </div>
    `;
}

function createTeacherCard(teacher) {
    const photoUrl = teacher.photo ? (window.location.origin + '/admin/public/storage/' + teacher.photo) : 'assets/img/default-teacher.png';
    
    return `
        <div class="bg-white rounded-2xl shadow-lg overflow-hidden border border-gray-200 hover:shadow-2xl hover:border-blue-300 transform hover:-translate-y-2 transition-all duration-300">
            <div class="bg-gradient-to-br from-blue-50 to-indigo-50 p-6 text-center">
                <div class="relative inline-block">
                    <div class="absolute inset-0 bg-blue-200/30 rounded-full blur-lg"></div>
                    <img src="${photoUrl}" 
                         alt="${teacher.name}" 
                         onerror="this.src='assets/img/default-teacher.png'"
                         class="relative w-24 h-24 rounded-full border-3 border-white shadow-xl object-cover mx-auto">
                </div>
                <h4 class="text-xl font-bold text-gray-800 mt-4 mb-1">${teacher.name}</h4>
                <p class="text-blue-600 font-semibold text-sm mb-1">${teacher.designation || 'Teacher'}</p>
                ${teacher.department ? `<p class="text-gray-500 text-xs">${teacher.department}</p>` : ''}
            </div>
            <div class="p-6">
                ${teacher.qualification ? `
                    <div class="mb-3">
                        <p class="text-xs font-semibold text-gray-700 mb-1 flex items-center">
                            <i class="fas fa-graduation-cap text-blue-600 mr-2"></i>Qualification
                        </p>
                        <p class="text-sm text-gray-600">${teacher.qualification}</p>
                    </div>
                ` : ''}
                ${teacher.specialization ? `
                    <div class="mb-3">
                        <p class="text-xs font-semibold text-gray-700 mb-1 flex items-center">
                            <i class="fas fa-book text-blue-600 mr-2"></i>Specialization
                        </p>
                        <p class="text-sm text-gray-600">${teacher.specialization}</p>
                    </div>
                ` : ''}
                <div class="flex flex-wrap gap-3 mt-4 pt-4 border-t border-gray-100">
                    ${teacher.email ? `
                        <a href="mailto:${teacher.email}" class="text-blue-600 hover:text-blue-800 transition text-sm" title="Email">
                            <i class="fas fa-envelope"></i>
                        </a>
                    ` : ''}
                    ${teacher.phone ? `
                        <a href="tel:${teacher.phone}" class="text-blue-600 hover:text-blue-800 transition text-sm" title="Phone">
                            <i class="fas fa-phone"></i>
                        </a>
                    ` : ''}
                </div>
            </div>
        </div>
    `;
}

