@extends('admin.layouts.app')

@section('title', 'Subject Rules Matrix')

@section('content')

    <!-- Version Switcher -->
    <div class="d-flex align-items-center gap-2">
        <label class="small fw-bold text-muted">Version:</label>
        <form action="{{ route('subjects.config.index') }}" method="GET">
            <select name="version_id" class="form-select form-select-sm border-primary" onchange="this.form.submit()" style="min-width: 200px;">
                @foreach($versions as $v)
                    @if(!$v->trashed())
                        <option value="{{ $v->id }}" {{ $currentVersion && $currentVersion->id == $v->id ? 'selected' : '' }}>
                            {{ $v->name }} {{ $v->is_current ? '(Current Active)' : '' }}
                        </option>
                    @endif
                @endforeach
            </select>
        </form>
        <button class="btn btn-sm btn-outline-secondary" data-bs-toggle="modal" data-bs-target="#versionManagerModal">
            <i class="bi bi-gear-fill"></i>
        </button>
    </div>
</div>

<!-- Manager Modal -->
<div class="modal fade" id="versionManagerModal" tabindex="-1">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content border-0 shadow-lg">
            <div class="modal-header bg-light border-0 pb-0">
                <h5 class="modal-title fw-bold text-dark"><i class="bi bi-clock-history me-2 text-primary"></i>Version History</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body p-0">
                <div class="alert alert-light border-0 small text-muted mb-0 px-4 py-3">
                    <i class="bi bi-info-circle me-1"></i> Managing versions allows you to restore previous configurations or safely remove old drafts.
                </div>
                <div class="table-responsive" style="max-height: 60vh;">
                    <table class="table table-hover align-middle mb-0">
                        <thead class="bg-white text-muted small text-uppercase sticky-top">
                            <tr>
                                <th class="ps-4 py-3">Version Name</th>
                                <th class="py-3">Status</th>
                                <th class="py-3 d-none d-md-table-cell">Created</th>
                                <th class="text-end pe-4 py-3">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="border-top-0">
                            @foreach($versions as $ver)
                            <tr class="{{ $ver->trashed() ? 'bg-danger bg-opacity-10 grayscale' : '' }}">
                                <td class="ps-4">
                                    <div class="fw-bold text-dark">{{ $ver->name }}</div>
                                    <!-- Mobile Date Show -->
                                    <div class="d-md-none small text-muted">{{ $ver->created_at->format('d M Y') }}</div>
                                    @if($ver->trashed()) <span class="badge bg-danger rounded-pill py-1 px-2 mt-1" style="font-size: 0.7rem;">Deleted</span> @endif
                                </td>
                                <td>
                                    @if($ver->is_current)
                                        <span class="badge bg-success bg-opacity-10 text-success border border-success rounded-pill px-3 py-2">
                                            <i class="bi bi-check-circle-fill me-1"></i> Active
                                        </span>
                                    @else
                                        <span class="badge bg-secondary bg-opacity-10 text-secondary border border-secondary rounded-pill px-3 py-2">
                                            Inactive
                                        </span>
                                    @endif
                                </td>
                                <td class="small text-muted d-none d-md-table-cell custom-font">
                                    {{ $ver->created_at->format('d M Y, h:i A') }}
                                </td>
                                <td class="text-end pe-4">
                                    @if($ver->trashed())
                                        <form action="{{ route('subjects.config.version.restore', $ver->id) }}" method="POST" class="d-inline">
                                            @csrf
                                            <button class="btn btn-sm btn-light text-success fw-bold shadow-sm border" title="Restore">
                                                <i class="bi bi-recycle"></i> <span class="d-none d-sm-inline">Restore</span>
                                            </button>
                                        </form>
                                    @else
                                        @if(!$ver->is_current)
                                            <div class="d-flex justify-content-end gap-1">
                                                <form action="{{ route('subjects.config.version.activate', $ver->id) }}" method="POST">
                                                    @csrf
                                                    <button class="btn btn-sm btn-primary shadow-sm" title="Make Active">
                                                        <i class="bi bi-power"></i> <span class="d-none d-sm-inline">Activate</span>
                                                    </button>
                                                </form>
                                                <form action="{{ route('subjects.config.version.delete', $ver->id) }}" method="POST" onsubmit="return confirm('Delete this version?');">
                                                    @csrf @method('DELETE')
                                                    <button class="btn btn-sm btn-outline-danger border-0" title="Delete"><i class="bi bi-trash"></i></button>
                                                </form>
                                            </div>
                                        @else
                                            <span class="text-muted small fst-italic">Current Version</span>
                                        @endif
                                    @endif
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
            <div class="modal-footer border-0 bg-light py-2">
                <button type="button" class="btn btn-sm btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

@if(!$currentVersion)
    <div class="alert alert-warning">No active academic version found. Please seed or create one.</div>
@else

<!-- MATRIX TABLE -->
<div class="card shadow-sm border-0 mb-5">
    <div class="card-header bg-white py-3">
        <h6 class="mb-0 fw-bold text-dark"><i class="bi bi-grid-3x3-gap me-2"></i>Subject Assignment Matrix</h6>
    </div>
    <div class="table-responsive">
        <table class="table table-hover align-middle mb-0 text-center">
            <thead class="bg-light text-uppercase small text-muted">
                <tr>
                    <th class="text-start ps-4" width="40%">Subject</th>
                    <th class="fw-bold text-primary bg-primary bg-opacity-10">Science</th>
                    <th class="fw-bold text-success bg-success bg-opacity-10">Commerce</th>
                    <th class="fw-bold text-danger bg-danger bg-opacity-10">Arts</th>
                </tr>
            </thead>
            <tbody>
                @foreach($subjects as $sub)
                <tr>
                    <td class="text-start ps-4">
                        <div class="fw-bold text-dark">{{ $sub->name }}</div>
                        <span class="badge bg-light text-secondary border font-monospace small">{{ $sub->code }}</span>
                        @if($sub->has_multiple_papers) 
                            <i class="bi bi-files text-primary ms-1" title="Two Papers"></i>
                        @endif
                    </td>

                    <!-- SCIENCE -->
                    <td class="bg-primary bg-opacity-10">
                        <div class="form-check form-check-inline">
                            <input class="form-check-input subject-toggle" type="checkbox" 
                                data-subject="{{ $sub->id }}" data-group="science"
                                {{ isset($matrix[$sub->id]['science']) ? 'checked' : '' }}>
                        </div>
                    </td>

                    <!-- COMMERCE -->
                    <td class="bg-success bg-opacity-10">
                        <div class="form-check form-check-inline">
                            <input class="form-check-input subject-toggle" type="checkbox" 
                                data-subject="{{ $sub->id }}" data-group="commerce"
                                {{ isset($matrix[$sub->id]['commerce']) ? 'checked' : '' }}>
                        </div>
                    </td>

                    <!-- ARTS -->
                    <td class="bg-danger bg-opacity-10">
                        <div class="form-check form-check-inline">
                            <input class="form-check-input subject-toggle" type="checkbox" 
                                data-subject="{{ $sub->id }}" data-group="arts"
                                {{ isset($matrix[$sub->id]['arts']) ? 'checked' : '' }}>
                        </div>
                    </td>
                    
                </tr>
                @endforeach
            </tbody>
        </table>
    </div>
</div>

<!-- OPTIONAL BUCKETS (4th Subject Rule) -->
<div class="row g-4">
    <div class="col-12"><h5 class="fw-bold text-muted border-bottom pb-2">Elective / 4th Subject Configuration</h5></div>
    
    @foreach(['science', 'commerce', 'arts'] as $group)
    @php
        $electives = $config[$group]['electives'];
        $hasRule = $electives->isNotEmpty();
    @endphp
    <div class="col-md-4">
        <div class="card h-100 border bg-light">
            <div class="card-header bg-white fw-bold text-uppercase d-flex justify-content-between align-items-center">
                {{ $group }}
                @if(!$hasRule)
                    <button class="btn btn-xs btn-primary text-white" data-bs-toggle="modal" data-bs-target="#createBucketModal" data-group="{{ $group }}">+ Set 4th Subject</button>
                @endif
            </div>
            <div class="card-body">
                @if($hasRule)
                    <div class="card mb-2 shadow-sm border-0">
                        <div class="card-body p-2">
                             <div class="d-flex justify-content-between align-items-start">
                                 <div>
                                     <strong class="text-primary" style="font-size: 13px;">4th Subject Rule</strong>
                                     <div class="small text-muted mb-1">Student chooses 1 from below:</div>
                                 </div>
                                 <div class="d-flex gap-1">
                                     <button class="btn btn-sm btn-light text-primary border-0" 
                                        data-bs-toggle="modal" 
                                        data-bs-target="#createBucketModal" 
                                        data-group="{{ $group }}"
                                        data-selected="{{ $electives->pluck('subject_id') }}"
                                        title="Edit Selection">
                                        <i class="bi bi-pencil-square fs-6"></i>
                                     </button>
                                     <form action="{{ route('subjects.config.delete-bucket', $group) }}" method="POST" onsubmit="return confirm('Remove 4th subject rule?');">
                                        @csrf @method('DELETE')
                                        <button class="btn btn-sm btn-light text-danger border-0" title="Remove Rule"><i class="bi bi-trash fs-6"></i></button>
                                     </form>
                                 </div>
                             </div>
                             
                             <div class="d-flex flex-wrap gap-1 mt-2">
                                @foreach($electives as $elective)
                                    <span class="badge bg-light text-dark border">{{ $elective->subject->name }}</span>
                                @endforeach
                             </div>
                        </div>
                    </div>
                @else
                    <div class="text-muted small text-center py-4">
                        No 4th subject rule set.<br>
                        <button class="btn btn-sm btn-link" data-bs-toggle="modal" data-bs-target="#createBucketModal" data-group="{{ $group }}">Click to Configure</button>
                    </div>
                @endif
            </div>
        </div>
    </div>
    @endforeach
</div>

<!-- Modal: Create Bucket -->
<div class="modal fade" id="createBucketModal" tabindex="-1">
    <div class="modal-dialog">
        <form action="{{ route('subjects.config.store-bucket') }}" method="POST">
            @csrf
            <input type="hidden" name="version_id" value="{{ $currentVersion->id }}">
            <input type="hidden" name="group_code" id="bucketGroupCode">
            <!-- Force 1 selection for 4th subject logic -->
            <input type="hidden" name="min_select" value="0">
            <input type="hidden" name="max_select" value="1">
            <input type="hidden" name="name" value="4th Subject"> <!-- Hardcoded Name -->
            
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Configure 4th Subject Options</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info small">
                        <strong>Instructions:</strong><br>
                        1. Select subjects below that students can choose as their 4th Subject.<br>
                        2. These subjects will become <strong>Optional</strong> (Student picks 1).<br>
                        3. All other subjects assigned in the Matrix will remain <strong>Mandatory</strong>.
                    </div>

                    <div class="item-wrapper border rounded p-2" style="max-height: 250px; overflow-y: auto;">
                        @foreach($subjects as $sub)
                            <div class="form-check subject-option" data-id="{{ $sub->id }}">
                                <input class="form-check-input bucket-chk" type="checkbox" name="subject_ids[]" value="{{ $sub->id }}" id="chk{{ $sub->id }}">
                                <label class="form-check-label" for="chk{{ $sub->id }}">{{ $sub->name }} ({{ $sub->code }})</label>
                            </div>
                        @endforeach
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </div>
        </form>
    </div>
</div>

@endif

@endsection

@push('scripts')
<!-- Toast Container -->
<div class="toast-container position-fixed bottom-0 end-0 p-3" style="z-index: 1100;">
  <div id="liveToast" class="toast align-items-center text-white bg-success border-0" role="alert" aria-live="assertive" aria-atomic="true">
    <div class="d-flex">
      <div class="toast-body" id="toastMessage">
        Configuration Saved.
      </div>
      <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
    </div>
  </div>
</div>

<script>
    // Toast Helper
    function showToast(message, type = 'success') {
        const toastEl = document.getElementById('liveToast');
        const toastBody = document.getElementById('toastMessage');
        const toast = new bootstrap.Toast(toastEl);
        
        toastEl.className = `toast align-items-center text-white bg-${type} border-0`;
        toastBody.textContent = message;
        toast.show();
    }

    // 1. Bucket Modal Data & Filtering
    const createBucketModal = document.getElementById('createBucketModal');
    if (createBucketModal) {
        createBucketModal.addEventListener('show.bs.modal', event => {
            const button = event.relatedTarget;
            const groupCode = button.getAttribute('data-group');
            // Parse existing selection if editing
            const selectedRaw = button.getAttribute('data-selected');
            const selectedIds = selectedRaw ? JSON.parse(selectedRaw) : [];

            document.getElementById('bucketGroupCode').value = groupCode;

            // Reset all checkboxes first
            document.querySelectorAll('.bucket-chk').forEach(chk => chk.checked = false);

            // Filter subjects: Only show those checked in the Main Matrix for this Group
            // 1. Find all checkboxes in the matrix for this group that are Checked
            const assignedIds = [];
            document.querySelectorAll(`.subject-toggle[data-group="${groupCode}"]:checked`).forEach(el => {
                assignedIds.push(el.dataset.subject);
            });

            // 2. Show/Hide and Check/Uncheck in Modal
            document.querySelectorAll('.subject-option').forEach(opt => {
                const subId = opt.getAttribute('data-id');
                const checkbox = opt.querySelector('.bucket-chk');
                
                if (assignedIds.includes(subId)) {
                   opt.classList.remove('d-none');
                   // If editing, check it if previously selected
                   if (selectedIds.includes(parseInt(subId))) {
                       checkbox.checked = true;
                   }
                } else {
                   opt.classList.add('d-none');
                }
            });
        });
    }

    // 2. AJAX Toggle Logic
    const versionId = "{{ $currentVersion->id ?? '' }}";

    document.querySelectorAll('.subject-toggle').forEach(checkbox => {
        checkbox.addEventListener('change', async function() {
            const subjectId = this.dataset.subject;
            const groupCode = this.dataset.group;
            const isActive = this.checked;
            
            // Visual Feedback immediate
            const originalState = !isActive; // to revert on fail

            fetch("{{ route('subjects.config.toggle-fixed') }}", {
                method: 'POST',
                headers: { 
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({
                    version_id: versionId,
                    group_code: groupCode,
                    subject_id: subjectId,
                    active: isActive
                })
            })
            .then(res => {
                if (!res.ok) {
                    return res.text().then(text => { throw new Error(text); });
                }
                return res.json();
            })
            .then(data => {
                if(!data.success) {
                    showToast('Error: ' + (data.message || 'Failed'), 'danger');
                    this.checked = originalState;
                } else {
                    if (data.version_changed) {
                        showToast('New Version Created! Reloading...', 'info');
                        setTimeout(() => window.location.reload(), 1500);
                    } else {
                        showToast('Saved Successfully');
                    }
                }
            })
            .catch(err => {
                console.error(err);
                showToast('System Error', 'danger');
                this.checked = originalState;
            });
        });
    });
</script>
@endpush
